/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.backoff.session;

import com.mulesoft.anypoint.backoff.configuration.BackoffConfiguration;

import java.util.List;

import com.mulesoft.anypoint.retry.session.ErrorSessionMetadata;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Process the {@link SessionMetadata} with a {@link BackoffConfiguration} to decide whether failure percentage threshold has been
 * reached and backoff should be applied.
 */
public class PlatformBackoffBarrier implements BackoffBarrier {

  private static final Logger LOGGER = LoggerFactory.getLogger(PlatformBackoffBarrier.class);

  @Override
  public boolean shouldBackoff(SessionMetadata sessionMetadata, BackoffConfiguration configuration) {
    int requests = sessionMetadata.requests();

    if (requests == 0) {
      return false;
    }

    int failures = failureCount(sessionMetadata, configuration.statusCodes());
    double failurePercentage = (double) failures / requests * 100;

    boolean hasFailed = failurePercentage >= configuration.failurePercentage();

    if (LOGGER.isDebugEnabled()) {
      String template = "Request failures {}% ({}/{}) {} threshold of {}%";
      if (failures > 0) {
        LOGGER.debug(template,
                     failurePercentage, failures, requests, thresholdRelation(hasFailed),
                     configuration.failurePercentage());
      } else {
        LOGGER.trace(template,
                     failurePercentage, failures, requests, thresholdRelation(hasFailed),
                     configuration.failurePercentage());
      }
    }

    return hasFailed;
  }

  @Override
  public boolean gotError(SessionMetadata sessionMetadata) {
    return sessionMetadata instanceof ErrorSessionMetadata;
  }

  private int failureCount(SessionMetadata sessionMetadata, List<Integer> backoffStatusCodes) {
    return backoffStatusCodes.stream()
        .mapToInt(sessionMetadata::getCount)
        .sum();
  }

  private String thresholdRelation(boolean hasFailed) {
    return hasFailed ? "above or equal to" : "below";
  }

}
