/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.backoff.state;


import static java.util.Optional.empty;
import static java.util.Optional.of;

import com.mulesoft.mule.runtime.gw.api.time.period.Period;
import com.mulesoft.anypoint.backoff.configuration.BackoffConfiguration;
import com.mulesoft.anypoint.backoff.session.BackoffBarrier;
import com.mulesoft.anypoint.backoff.session.SessionMetadata;

import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Defines a Backoff state, that can evolve into another one depending on the {@link BackoffBarrier}. Each {@link State} knows the
 * delay it represents.
 * 
 * Also, it can be configured ro run a closure once depending on its {@link State}: {@link Stable}/{@link Unstable}.
 */
public abstract class State {

  protected static final Logger LOGGER = LoggerFactory.getLogger(State.class);

  protected BackoffConfiguration configuration;
  protected BackoffBarrier backoffBarrier;
  protected Optional<Runnable> unstableClosure;
  protected Optional<Runnable> stableClosure;
  protected Optional<Runnable> errorClosure;

  public State(BackoffConfiguration configuration, BackoffBarrier backoffBarrier) {
    this.configuration = configuration;
    this.backoffBarrier = backoffBarrier;
    resetClosures();
  }

  /**
   * Defines the next state given a {@link SessionMetadata}.
   * 
   * @param sessionMetadata last {@link SessionMetadata}.
   * @return the next state
   */
  public abstract State next(SessionMetadata sessionMetadata);

  /**
   * @return the {@link Period} of delay it represents.
   */
  public abstract Period delay();

  /**
   * Configures a closure to be executed only once if this {@link State} is {@link Unstable}. If not configured, and empty closure
   * is defined.
   * 
   * @param closure to execute.
   * @return this.
   */
  public State ifUnstable(Runnable closure) {
    this.unstableClosure = of(closure);
    return this;
  }

  /**
   * Configures a closure to be executed only once if this {@link State} is {@link Error}. If not configured, and empty closure is
   * defined.
   *
   * @param closure to execute.
   * @return this.
   */
  public State ifError(Runnable closure) {
    this.errorClosure = of(closure);
    return this;
  }

  /**
   * Configures a closure to be executed only once if this {@link State} is {@link Stable}. If not configured, and empty closure
   * is defined.
   * 
   * @param closure to execute.
   * @return this.
   */
  public State otherwise(Runnable closure) {
    this.stableClosure = of(closure);
    return this;
  }

  /**
   * Executes the configured closure.
   * 
   * @return this.
   */
  public abstract State go();

  /**
   * Logs current state.
   *
   * @return this.
   */
  public abstract State log();

  protected State resetClosures() {
    this.unstableClosure = empty();
    this.stableClosure = empty();
    return this;
  }
}


