/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.retry;

import static com.mulesoft.mule.runtime.gw.api.time.period.Period.millis;
import static com.mulesoft.mule.runtime.gw.api.time.period.Period.seconds;
import static com.mulesoft.anypoint.backoff.scheduler.configuration.SchedulingConfiguration.configuration;
import static java.util.Optional.ofNullable;
import static java.util.concurrent.Executors.newScheduledThreadPool;

import org.mule.runtime.core.api.util.concurrent.NamedThreadFactory;

import com.mulesoft.anypoint.backoff.configuration.BackoffConfiguration;
import com.mulesoft.anypoint.backoff.configuration.BackoffConfigurationSupplier;
import com.mulesoft.anypoint.backoff.scheduler.BackoffScheduler;
import com.mulesoft.anypoint.backoff.scheduler.configuration.FastRecoveryConfiguration;
import com.mulesoft.anypoint.backoff.scheduler.configuration.SchedulingConfiguration;
import com.mulesoft.anypoint.backoff.scheduler.factory.BackoffSchedulerFactory;
import com.mulesoft.anypoint.backoff.scheduler.observer.FastRecoveryObserver;
import com.mulesoft.anypoint.backoff.scheduler.runnable.BackoffRunnable;
import com.mulesoft.anypoint.backoff.scheduler.runnable.FastRecovery;
import com.mulesoft.anypoint.backoff.state.Stable;
import com.mulesoft.anypoint.backoff.state.Unstable;
import com.mulesoft.anypoint.retry.barrier.BackoffRetrierBarrier;
import com.mulesoft.anypoint.retry.barrier.BackoffWhileRetryFails;
import com.mulesoft.anypoint.retry.barrier.BackoffWhilstAlone;
import com.mulesoft.anypoint.retry.exception.RunnableRetrierException;
import com.mulesoft.anypoint.retry.runnable.RetrierRunnable;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Executes multiple {@link Runnable Runnables} in a separate thread until they complete successfully. An execution is considered
 * successful when it does not end with an {@link Exception}.
 * <p/>
 * Under the hood, this object uses the {@link BackoffScheduler}'s engine, in {@link FastRecovery} mode. This means that backoff
 * will be applied to those task who raise an {@link Exception}. The first execution that finishes without an error, will mean
 * that it ended successfully and will not be rescheduled again. {@link BackoffConfiguration} can be set to each instance of this.
 * <p/>
 * {@link BackoffRunnableRetrier} handles one queue per instance of key (of type {@link T}). Subsequent queued {@link Runnable
 * Runnables} for a key will not be executed until its current scheduled task reaches the {@link Stable} state. NOTE: if for a
 * certain key, running task never halts, the remaining queued tasks will starve.
 * <p/>
 * To minimize runtime resource consumption, {@link BackoffScheduler} will be disposed if there are no further tasks to process.
 * It will be recreated as soon as a {@link Runnable} is scheduled.
 */
public class BackoffRunnableRetrier<T> implements RunnableRetrier<T>, FastRecoveryObserver {

  private static final Logger LOGGER = LoggerFactory.getLogger(BackoffRunnableRetrier.class);

  private final BackoffConfiguration configuration;
  private final BackoffSchedulerFactory schedulerFactory;
  private final SchedulingConfiguration initialSchedulingConfiguration;

  private final Lock lock = new ReentrantLock();
  private final Map<T, List<Runnable>> runnables = new HashMap<>();
  private final String threadName;

  private BackoffRetrierBarrier barrier;
  private BackoffScheduler scheduler;

  private BackoffRunnableRetrier(String threadName,
                                 BackoffConfiguration configuration,
                                 BackoffSchedulerFactory schedulerFactory,
                                 BackoffRetrierBarrier barrier,
                                 SchedulingConfiguration initialSchedulingConfiguration) {
    this.barrier = barrier;
    this.threadName = threadName;
    this.configuration = configuration;
    this.schedulerFactory = schedulerFactory;
    this.initialSchedulingConfiguration = initialSchedulingConfiguration;

    checkFastRecovery(this.configuration);
  }

  /**
   * {@inheritDoc}
   *
   * If there is no {@link Runnable} already running for that key, task will be immediately scheduled as a
   * {@link BackoffRunnable}. Otherwise, it will be queued for later processing.
   */
  @Override
  public RunnableRetrier<T> scheduleRetry(T key, Runnable runnable) {
    atomically(() -> {
      createScheduler();

      add(key, runnable);

      try {
        if (thereIsOnlyARunnable(key)) {
          schedule(key, runnable);
        } else {
          LOGGER.trace("There is a runnable running for key {}. Runnable {} will be queued.", key, runnable.hashCode());
        }
      } catch (Throwable t) {
        runnables.get(key).remove(runnable);
      }
    });
    return this;
  }

  @Override
  public boolean hasQueuedRunnables(T key) {
    return !thereIsOnlyARunnable(key);
  }

  /**
   * Return whether the retrier is idle or there are tasks running or pending for execution for a specific key.
   *
   * @param key pending task's key.
   * @return true if it is idle, false otherwise.
   */
  public boolean isIdle(T key) {
    return !runnablesPending(key);
  }

  /**
   * {@inheritDoc}
   * 
   * While {@link BackoffRunnable} remains {@link Unstable}, this object won't do anything. The {@link BackoffScheduler} will
   * automatically reschedule the task.
   */
  @Override
  public FastRecoveryObserver fastRecoveryUnstable(BackoffRunnable runnable, FastRecoveryConfiguration configuration) {
    LOGGER.trace("BackoffRunnable {} remains unstable, it will be retried with configuration {}.", runnable, configuration);
    return this;
  }

  /**
   * {@inheritDoc}
   *
   * When {@link BackoffRunnable} returns to {@link Stable} state, the {@link BackoffScheduler} will remove the
   * {@link FastRecovery} task. This object then, must schedule the next queued {@link Runnable} for that key, or dispose the
   * scheduler if it was the last one.
   */
  @Override
  public FastRecoveryObserver fastRecoveryStable(BackoffRunnable backoffRunnable, FastRecoveryConfiguration configuration) {
    atomically(() -> {
      Runnable runnable = asRetrier(backoffRunnable).inner();
      T key = asRetrier(backoffRunnable).key();
      runnables.get(key).remove(runnable);
      LOGGER.trace("BackoffRunnable {} is now stable, it will be removed. Inner runnable {} has been dropped.", backoffRunnable,
                   runnable.hashCode());

      if (!runnablesPending()) {
        dispose();
      } else if (runnablesPending(key)) {
        schedule(key, runnables.get(key).get(0));
      }
    });
    return this;
  }

  /**
   * {@inheritDoc}
   *
   * When {@link BackoffRunnable} goes to {@link Error} state, the {@link BackoffScheduler} will remove the {@link FastRecovery}
   * task and log the error. This object then, must schedule the next queued {@link Runnable} for that key, or dispose the
   * scheduler if it was the last one.
   */
  @Override
  public FastRecoveryObserver fastRecoveryAbort(BackoffRunnable backoffRunnable) {
    atomically(() -> {
      Runnable runnable = asRetrier(backoffRunnable).inner();
      T key = asRetrier(backoffRunnable).key();
      runnables.get(key).remove(runnable);
      LOGGER.trace("BackoffRunnable {} finished with error, it will be dropped and removed from the queue.", backoffRunnable);

      if (!runnablesPending()) {
        dispose();
      } else if (runnablesPending(key)) {
        schedule(key, runnables.get(key).get(0));
      }
    });
    return this;
  }

  @Override
  public void dispose() {
    if (scheduler != null) {
      LOGGER.trace("Disposing BackoffScheduler {}", scheduler.hashCode());
      scheduler.dispose();
      scheduler = null;
    }
    runnables.clear();
  }

  public static class Builder<T> implements com.mulesoft.mule.runtime.gw.api.construction.Builder<BackoffRunnableRetrier<T>> {

    private final String threadName;
    private BackoffConfigurationSupplier configurationSupplier;
    private BackoffRetrierBarrier<T> retrierBarrier;
    private BackoffSchedulerFactory schedulerFactory;
    private final List<Integer> statusCodes;
    private SchedulingConfiguration initialSchedulingConfiguration;
    private final boolean backoffEnabled;

    public Builder(String retrierName,
                   List<Integer> outagesStatusCodes,
                   boolean backoffEnabled) {
      this.threadName = retrierName;
      this.configurationSupplier = new BackoffConfigurationSupplier();
      this.retrierBarrier = new BackoffWhileRetryFails();
      this.statusCodes = outagesStatusCodes;
      this.backoffEnabled = backoffEnabled;
    }

    public Builder<T> retryUntilNewSchedule() {
      this.retrierBarrier = new BackoffWhilstAlone<>();
      return this;
    }

    public Builder<T> configurationSupplier(BackoffConfigurationSupplier configurationSupplier) {
      this.configurationSupplier = configurationSupplier;
      return this;
    }

    public Builder<T> scheduler(BackoffSchedulerFactory schedulerFactory,
                                SchedulingConfiguration configuration) {
      this.schedulerFactory = schedulerFactory;
      this.initialSchedulingConfiguration = configuration;
      return this;
    }

    @Override
    public BackoffRunnableRetrier<T> build() {
      return new BackoffRunnableRetrier(threadName,
                                        configurationSupplier.forScheduleOnce(statusCodes, backoffEnabled),
                                        schedulerFactory,
                                        retrierBarrier,
                                        initialSchedulingConfiguration);
    }
  }

  private BackoffRunnable backoffRunnable(T key, Runnable runnable) {
    barrier.initialise(key, this);
    return new RetrierRunnable(key, runnable, configuration, barrier);
  }

  private void add(T key, Runnable runnable) {
    if (!runnables.containsKey(key)) {
      runnables.put(key, new ArrayList<>());
    }

    runnables.get(key).add(runnable);
  }

  private void schedule(T key, Runnable runnable) {
    BackoffRunnable backoffRunnable = backoffRunnable(key, runnable);
    LOGGER.trace("Scheduling runnable {} in BackoffRunnable {} with configuration {}.",
                 runnable.hashCode(),
                 backoffRunnable,
                 initialSchedulingConfiguration);
    scheduler.schedule(backoffRunnable, initialSchedulingConfiguration, this);
  }

  private void createScheduler() {
    if (scheduler == null) {
      scheduler = schedulerFactory.create(newScheduledThreadPool(1,
                                                                 new NamedThreadFactory(threadName)));
      LOGGER.trace("BackoffScheduler {} created using factory {}. Thread pool will be {}.",
                   scheduler.hashCode(), schedulerFactory.getClass().getSimpleName(), threadName);
    }
  }

  private boolean runnablesPending() {
    return runnables.keySet().stream().anyMatch(this::runnablesPending);
  }

  private boolean runnablesPending(T key) {
    return !ofNullable(runnables.get(key)).map(List::isEmpty).orElse(true);
  }

  private boolean thereIsOnlyARunnable(T key) {
    return runnables.get(key).size() == 1;
  }

  private void atomically(Runnable closure) {
    lock.lock();
    try {
      closure.run();
    } finally {
      lock.unlock();
    }
  }

  private RetrierRunnable<T> asRetrier(BackoffRunnable runnable) {
    return (RetrierRunnable<T>) runnable;
  }

  private void checkFastRecovery(BackoffConfiguration backoffConfiguration) {
    if (!backoffConfiguration.isFastRecovery()) {
      throw new RunnableRetrierException("Backoff configuration should be one of fast recovery.");
    }
  }

  public static SchedulingConfiguration delayInitialScheduling(int initialDelay) {
    return configuration(seconds(initialDelay));
  }

  public static SchedulingConfiguration zeroDelayOnScheduling() {
    return configuration(millis(0));
  }
}
