/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.client.httpclient.interceptors;

import static com.mulesoft.mule.runtime.gw.api.time.DateTime.now;
import static com.mulesoft.mule.runtime.gw.client.httpclient.interceptors.AccessTokens.invalidTokens;

import com.mulesoft.mule.runtime.gw.api.time.DateTime;

import com.google.gson.JsonObject;
import com.google.gson.JsonParser;

import java.util.Base64;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Function;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AccessTokenManager {

  private static final Logger LOGGER = LoggerFactory.getLogger(AccessTokenManager.class);

  private static final String EXP_CLAIM = "exp";
  private static final String IAT_CLAIM = "iat";

  private AccessTokens token;
  private final ReentrantReadWriteLock accessTokenLock;

  AccessTokenManager() {
    this.accessTokenLock = new ReentrantReadWriteLock();
    this.token = invalidTokens();
  }

  /**
   * @param coreServicesTokenProvider a supplier for the core services token, it will only be called if necessary.
   * @param platformTokenProvider a function that receives the core services token and obtains the platform token from exchange,
   *                            it will only be called if necessary.
   * @return the current platform tokens or a new set of tokens if they expired or are close to expiration.
   */
  public AccessTokens getUpdatedTokens(Supplier<String> coreServicesTokenProvider,
                                       Function<String, String> platformTokenProvider) {
    try {
      accessTokenLock.writeLock().lock();

      if (token.haveExpired()) {
        LOGGER.debug("The platform token has expired, requesting a new one.");
        invalidateToken();
      }

      String coreServicesToken = token.getCoreServicesToken();
      String platformAccessToken = token.getPlatformToken();

      if (coreServicesToken == null) {
        coreServicesToken = coreServicesTokenProvider.get();
        platformAccessToken = null; // force portal token exchange
        token = new AccessTokens(coreServicesToken);
      }

      if (platformAccessToken == null) {
        platformAccessToken = platformTokenProvider.apply(coreServicesToken);
        token = new AccessTokens(coreServicesToken, platformAccessToken, calculateExpiration(platformAccessToken));
      }

      return token;
    } catch (Exception e) {
      invalidateToken();
      throw e;
    } finally {
      accessTokenLock.writeLock().unlock();
    }
  }

  DateTime calculateExpiration(String platformToken) {
    if (platformToken != null) {
      String[] tokenParts = platformToken.split("\\.");
      if (tokenParts.length >= 2) {
        try {
          JsonObject tokenBody = new JsonParser()
              .parse(new String(Base64.getMimeDecoder().decode(tokenParts[1])))
              .getAsJsonObject();
          return now()
              .plusMillis((int) (tokenBody.get(EXP_CLAIM).getAsLong() - tokenBody.get(IAT_CLAIM).getAsLong() - 120) * 1000);
        } catch (Exception e) {
          LOGGER.debug("Error parsing the jwt: {}.", e.getMessage());
        }
      }
    }
    LOGGER.debug("The expiration information of the Json Web Token, from platform services, could not be extracted.");
    return null;
  }

  public void reset() {
    accessTokenLock.writeLock().lock();
    try {
      invalidateToken();
    } finally {
      accessTokenLock.writeLock().unlock();
    }
  }

  /**
   * Invalidate the current value of the token if the token has still the value accesTokenInUse.
   * This is necessary to avoid unnecessary invalidations of old tokens due to multiple threads using the same TokenManager.
   * @param accessTokenInUse the value that the core services token must have to invalidate this token.
   */
  public void resetIfEquals(String accessTokenInUse) {
    try {
      accessTokenLock.writeLock().lock();
      if (accessTokenInUse != null && token != null && accessTokenInUse.equals(token.getCoreServicesToken())) {
        invalidateToken();
      }
    } finally {
      accessTokenLock.writeLock().unlock();
    }
  }

  private void invalidateToken() {
    this.token = invalidTokens();
  }
}
