/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.client.model;

import com.mulesoft.mule.runtime.gw.api.analytics.AnalyticsHttpEvent;
import com.mulesoft.mule.runtime.gw.api.analytics.DeploymentType;
import com.mulesoft.mule.runtime.gw.api.analytics.RequestDisposition;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonView;

import java.io.Serializable;

/**
 * This Class is a DTO to send the analytics information to Cloud Analytics.
 */
@JsonAutoDetect
public class HttpEvent implements Serializable {

  private static final long serialVersionUID = 8947695862352979432L;

  @JsonProperty("api_id")
  private Integer apiId;

  @JsonProperty("api_version_id")
  private Long apiVersionId;

  @JsonProperty("org_id")
  private String orgId;

  @JsonProperty("host_id")
  private String hostId;

  @JsonProperty("client_id")
  private String clientId;

  @JsonProperty("transaction_id")
  private String transactionId;

  @JsonProperty("received_ts")
  private String receivedTs;

  @JsonProperty("replied_ts")
  private String repliedTs;

  @JsonProperty("client_ip")
  private String clientIp;

  @JsonProperty("verb")
  private String verb;

  @JsonProperty("path")
  private String path;

  @JsonProperty("status_code")
  private int statusCode;

  @JsonProperty("user_agent")
  private String userAgent;

  @JsonProperty("request_bytes")
  private int requestBytes;

  @JsonProperty("response_bytes")
  private int responseBytes;

  @JsonProperty("request_disposition")
  private RequestDisposition requestDisposition;

  @JsonProperty("policy_violation")
  private PolicyViolationDto policyViolation;

  @JsonProperty("api_name")
  @JsonView(HttpEventViews.ExternalAnalytics.class)
  private String apiName;

  @JsonProperty("api_version")
  @JsonView(HttpEventViews.ExternalAnalytics.class)
  private String apiVersion;

  @JsonProperty("instance_name")
  @JsonView(HttpEventViews.ExternalAnalytics.class)
  private String instanceName;

  @JsonProperty("application_name")
  @JsonView(HttpEventViews.ExternalAnalytics.class)
  private String applicationName;

  @JsonProperty("event_id")
  private String eventId;

  @JsonProperty("deployment_type")
  private DeploymentType deploymentType;

  public HttpEvent(Integer apiId, Long apiVersionId, String orgId, String hostId, String clientId, String transactionId,
                   String receivedTs, String repliedTs, String clientIp, String verb, String path, int statusCode,
                   String userAgent, int requestBytes, int responseBytes, RequestDisposition requestDisposition,
                   PolicyViolationDto policyViolation, String apiName, String apiVersion, String instanceName,
                   String applicationName, String eventId, DeploymentType deploymentType) {
    this.apiId = apiId;
    this.apiVersionId = apiVersionId;
    this.orgId = orgId;
    this.hostId = hostId;
    this.clientId = clientId;
    this.transactionId = transactionId;
    this.receivedTs = receivedTs;
    this.repliedTs = repliedTs;
    this.clientIp = clientIp;
    this.verb = verb;
    this.path = path;
    this.statusCode = statusCode;
    this.userAgent = userAgent;
    this.requestBytes = requestBytes;
    this.responseBytes = responseBytes;
    this.requestDisposition = requestDisposition;
    this.policyViolation = policyViolation;
    this.apiName = apiName;
    this.apiVersion = apiVersion;
    this.instanceName = instanceName;
    this.applicationName = applicationName;
    this.eventId = eventId;
    this.deploymentType = deploymentType;
  }

  /**
   * <p>
   * Note: The returned HttpEvent does not contain information about the muleAppName nor the flowName. Since those fields are
   * not required by Cloud Analytics.
   * </p>
   * @param event The AnalyticsHttpEvent event from which all the information.
   * @return an instance of the HttpEvent that contains the information from the event.
   */
  public static HttpEvent from(AnalyticsHttpEvent event) {
    PolicyViolationDto policyViolationDto =
        event.getPolicyViolation() != null ? PolicyViolationDto.from(event.getPolicyViolation()) : null;
    return new HttpEvent(event.getApiId(), event.getApiVersionId(), event.getOrgId(), event.getHostId(), event.getClientId(),
                         event.getTransactionId(), event.getReceivedTs(), event.getRepliedTs(), event.getClientIp(),
                         event.getVerb(), event.getPath(), event.getStatusCode(), event.getUserAgent(), event.getRequestBytes(),
                         event.getResponseBytes(), event.getRequestDisposition(), policyViolationDto, event.getApiName(),
                         event.getApiVersion(), event.getInstanceName(), event.getApplicationName(), event.getEventId(),
                         event.getDeploymentType());
  }

  public Integer getApiId() {
    return apiId;
  }

  public Long getApiVersionId() {
    return apiVersionId;
  }

  public String getOrgId() {
    return orgId;
  }

  public String getHostId() {
    return hostId;
  }

  public String getClientId() {
    return clientId;
  }

  public String getTransactionId() {
    return transactionId;
  }

  public String getReceivedTs() {
    return receivedTs;
  }

  public String getRepliedTs() {
    return repliedTs;
  }

  public String getClientIp() {
    return clientIp;
  }

  public String getVerb() {
    return verb;
  }

  public String getPath() {
    return path;
  }

  public int getStatusCode() {
    return statusCode;
  }

  public String getUserAgent() {
    return userAgent;
  }

  public int getRequestBytes() {
    return requestBytes;
  }

  public int getResponseBytes() {
    return responseBytes;
  }

  public RequestDisposition getRequestDisposition() {
    return requestDisposition;
  }

  public PolicyViolationDto getPolicyViolation() {
    return policyViolation;
  }

  public String getApiName() {
    return apiName;
  }

  public String getApiVersion() {
    return apiVersion;
  }

  public String getApplicationName() {
    return applicationName;
  }

  public String getEventId() {
    return eventId;
  }

  public String getInstanceName() {
    return instanceName;
  }

  public DeploymentType getDeploymentType() {
    return deploymentType;
  }

  @Override
  public String toString() {
    return "HttpEvent #" + hashCode() + "{" +
        "apiName=" + apiName +
        ", apiVersion=" + apiVersion +
        ", instanceName=" + instanceName +
        ", orgId='" + orgId + '\'' +
        ", hostId='" + hostId + '\'' +
        ", clientId='" + clientId + '\'' +
        ", applicationName='" + applicationName + '\'' +
        ", transactionId='" + transactionId + '\'' +
        ", receivedTs='" + receivedTs + '\'' +
        ", repliedTs='" + repliedTs + '\'' +
        ", clientIp='" + clientIp + '\'' +
        ", verb='" + verb + '\'' +
        ", path='" + path + '\'' +
        ", statusCode=" + statusCode +
        ", userAgent='" + userAgent + '\'' +
        ", requestBytes=" + requestBytes +
        ", responseBytes=" + responseBytes +
        ", requestDisposition=" + requestDisposition +
        ", policyViolation=" + policyViolation +
        ", eventId=" + eventId +
        ", deploymentType=" + deploymentType +
        '}';
  }

}
