/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.client.provider;

import static com.mulesoft.mule.runtime.gw.api.logging.ExceptionDescriptor.errorMessage;
import static org.apache.commons.lang3.StringUtils.isBlank;

import java.net.URISyntaxException;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import org.slf4j.Logger;

import com.mulesoft.mule.runtime.gw.api.config.GatewayConfiguration;
import com.mulesoft.mule.runtime.gw.api.exception.GatewayConfigurationException;
import com.mulesoft.mule.runtime.gw.client.ApiPlatformClient;
import com.mulesoft.mule.runtime.gw.client.exception.NormalizedExceptionMessageLogger;
import com.mulesoft.mule.runtime.gw.client.exception.RecoverableExceptionMessageLogger;
import com.mulesoft.mule.runtime.gw.client.exception.UnauthorizedException;
import com.mulesoft.mule.runtime.gw.logging.GatewayMuleAppLoggerFactory;
import com.mulesoft.mule.runtime.gw.retry.RunnableRetrier;

public class ApiPlatformClientProvider {

  private static final String CONNECTION_ERROR_DESCRIPTION = "connect with API Manager";
  private static final Logger LOGGER = GatewayMuleAppLoggerFactory.getLogger(ApiPlatformClientProvider.class);

  private final NormalizedExceptionMessageLogger normalizedLogger = new RecoverableExceptionMessageLogger(LOGGER);

  private final ApiPlatformClient client;

  private final RunnableRetrier<String> retrier;

  private List<ApiPlatformClientConnectionListener> connectionListeners = new CopyOnWriteArrayList<>();

  public ApiPlatformClientProvider(RunnableRetrier<String> reconnectionRetrier) {
    this.client = new ApiPlatformClient();
    this.retrier = reconnectionRetrier;
  }

  public boolean configureClient(GatewayConfiguration configuration) {
    try {
      if (validConfiguration(configuration)) {
        client.configure(configuration);
        return true;
      }
    } catch (GatewayConfigurationException e) {
      LOGGER.error("Unable to start API Platform Client. {}", errorMessage(e));
    } catch (URISyntaxException e) {
      LOGGER.error("Invalid URL configured. API Platform client is DISABLED. {}", errorMessage(e));
    } catch (Exception e) {
      LOGGER.error("An invalid Gateway configuration was detected. API Platform client is DISABLED. {}", errorMessage(e));
    }

    return false;
  }

  public void connectClient() {
    try {
      doConnect();
    } catch (Throwable t) {
      retrier.scheduleRetry("clientConnect", this::doConnect);
    }
  }

  private void doConnect() {
    try {
      client.connect();
      LOGGER.info("Client ID and Client Secret successfully validated against API Manager.");
      notifyListeners();
    } catch (UnauthorizedException e) {
      LOGGER.error("Client ID and Client Secret could not be validated against API Manager.");
      normalizedLogger.error(CONNECTION_ERROR_DESCRIPTION, e);
      throw e;
    } catch (Throwable t) {
      // This might mean that either the platform is down or the GW is having connection issues
      // Exception is not rethrown to allow reconnection when possible.
      normalizedLogger.error(CONNECTION_ERROR_DESCRIPTION, t);
      throw t;
    }
  }

  public ApiPlatformClient getClient() {
    return client;
  }

  public void shutdown() {
    client.shutdown();
    retrier.dispose();
  }

  public void addConnectionListener(ApiPlatformClientConnectionListener listener) {
    connectionListeners.add(listener);
  }

  private void notifyListeners() {
    connectionListeners.forEach(listener -> {
      try {
        listener.onClientConnected();
      } catch (Exception e) {
        LOGGER.warn("Error on Platform Client connected listener. {}", errorMessage(e));
      }
    });
  }

  private boolean validConfiguration(GatewayConfiguration configuration) {
    if (!platformUriIsSet(configuration)) {
      if (configuration.platformClient().isOnPrem()) {
        LOGGER.warn("On Prem mode is enabled but anypoint.platform.base_uri is not set. API Platform client is DISABLED.");
      } else {
        LOGGER.warn("anypoint.platform.base_uri is not set. API Platform client is DISABLED.");
      }
      return false;
    } else if (configuration.platformClient().isOfflineModeEnabled()) {
      LOGGER.warn("Client ID or Client Secret were not provided. API Platform client is DISABLED.");
      return false;
    }

    return true;
  }

  private boolean platformUriIsSet(GatewayConfiguration configuration) {
    return !isBlank(configuration.platformClient().getPlatformUri());
  }

}
