/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.client;

import static com.mulesoft.anypoint.tests.http.SimpleHttpServerResponse.builder;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideVariable;
import static java.lang.String.format;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertThat;
import static org.junit.rules.ExpectedException.none;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.RETURNS_DEEP_STUBS;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import java.io.IOException;
import java.io.StringWriter;

import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

import org.mule.tck.junit4.AbstractMuleTestCase;

import com.fasterxml.jackson.databind.ObjectMapper;

import com.mulesoft.anypoint.tests.rules.HttpServerRule;
import com.mulesoft.mule.runtime.gw.api.config.GatewayConfiguration;
import com.mulesoft.mule.runtime.gw.client.model.Me;
import com.mulesoft.mule.runtime.gw.client.model.Organization;
import com.mulesoft.mule.runtime.gw.client.model.OrganizationClient;
import com.mulesoft.mule.runtime.gw.client.model.User;
import com.mulesoft.mule.runtime.gw.client.provider.ApiPlatformClientConnectionListener;
import com.mulesoft.mule.runtime.gw.client.provider.ApiPlatformClientProvider;
import com.mulesoft.mule.runtime.gw.retry.BackoffRunnableRetrierFactory;
import com.mulesoft.mule.runtime.gw.retry.RunnableRetrier;

public class ApiPlatformClientProviderTestCase extends AbstractMuleTestCase {

  private static final String URI = "http://localhost:%s";
  private static final String CLIENT = "client";

  @Rule
  public HttpServerRule platformServer = new HttpServerRule("platformPort");

  @Rule
  public HttpServerRule coreServicesServer = new HttpServerRule("csPort");

  @Rule
  public ExpectedException expectedException = none();

  private ApiPlatformClientProvider provider;

  private GatewayConfiguration configuration;

  @Before
  public void setUp() {
    provider = new ApiPlatformClientProvider(retrier());

    configuration = mock(GatewayConfiguration.class, RETURNS_DEEP_STUBS);
  }

  @After
  public void tearDown() {
    provider.shutdown();
  }

  @Test
  public void initialiseRestClient() throws IOException {
    mockValidConfiguration();

    boolean configured = provider.configureClient(configuration);
    provider.connectClient();

    assertThat(configured, is(true));
    assertThat(provider.getClient().isConnected(), is(true));
  }

  @Test
  public void initialiseRestClientNotifiesListener() throws IOException {
    mockValidConfiguration();
    ApiPlatformClientConnectionListener listener = mock(ApiPlatformClientConnectionListener.class);
    provider.addConnectionListener(listener);

    boolean configured = provider.configureClient(configuration);
    provider.connectClient();

    assertThat(configured, is(true));
    assertThat(provider.getClient().isConnected(), is(true));
    verify(listener).onClientConnected();
  }

  @Test
  public void exceptionOnListenerDoesNotBlockOtherListeners() throws IOException {
    mockValidConfiguration();
    ApiPlatformClientConnectionListener listener1 = mock(ApiPlatformClientConnectionListener.class);
    ApiPlatformClientConnectionListener listener2 = mock(ApiPlatformClientConnectionListener.class);
    doThrow(new RuntimeException()).when(listener1).onClientConnected();
    provider.addConnectionListener(listener1);
    provider.addConnectionListener(listener2);

    boolean configured = provider.configureClient(configuration);
    provider.connectClient();

    assertThat(configured, is(true));
    assertThat(provider.getClient().isConnected(), is(true));
    verify(listener1).onClientConnected();
    verify(listener2).onClientConnected();
  }

  @Test
  public void initialiseRestClientTwice() throws IOException {
    mockValidConfiguration();

    boolean configured = provider.configureClient(configuration);
    provider.connectClient();
    provider.connectClient();

    assertThat(configured, is(true));
    assertThat(provider.getClient().isConnected(), is(true));
  }

  @Test
  public void badConfigOfflineModeEnabled() throws IOException {
    when(configuration.platformClient().getPlatformUri()).thenReturn(platformUri());
    when(configuration.platformClient().isOfflineModeEnabled()).thenReturn(true);

    boolean configured = provider.configureClient(configuration);

    assertThat(configured, is(false));
    assertThat(provider.getClient().isConnected(), is(false));
  }

  @Test
  public void badConfigEmptyPlatformUrl() {
    when(configuration.platformClient().getClientId()).thenReturn(CLIENT);
    when(configuration.platformClient().getClientSecret()).thenReturn(CLIENT);

    boolean configured = provider.configureClient(configuration);

    assertThat(configured, is(false));
    assertThat(provider.getClient().isConnected(), is(false));
  }

  @Test
  public void badConfigUriSyntaxException() {
    when(configuration.platformClient().getPlatformUri()).thenReturn("invalid@:malformedUrl");

    boolean configured = provider.configureClient(configuration);

    assertThat(configured, is(false));
    assertThat(provider.getClient().isConnected(), is(false));
  }

  @Test
  public void badConnectionCoreServices401RetrierStarted() throws IOException {
    RunnableRetrier runnableRetrier = mock(RunnableRetrier.class);
    overrideVariable("retrier").in(provider).with(runnableRetrier);
    coreServicesServer.getHttpServer().setResponse(builder().statusCode(401).build());
    when(configuration.platformClient().getPlatformUri()).thenReturn(platformUri());
    when(configuration.platformClient().getClientId()).thenReturn(CLIENT);
    when(configuration.platformClient().getClientSecret()).thenReturn(CLIENT);

    provider.configureClient(configuration);
    provider.connectClient();

    verify(runnableRetrier).scheduleRetry(any(), any());
    assertThat(provider.getClient().isConnected(), is(false));
  }

  @Test
  public void badConnectionPlatform500RetrierStarted() {
    RunnableRetrier runnableRetrier = mock(RunnableRetrier.class);
    overrideVariable("retrier").in(provider).with(runnableRetrier);
    platformServer.getHttpServer().setResponse(builder().statusCode(500).build());
    when(configuration.platformClient().getPlatformUri()).thenReturn(platformUri());
    when(configuration.platformClient().getClientId()).thenReturn(CLIENT);
    when(configuration.platformClient().getClientSecret()).thenReturn(CLIENT);

    provider.configureClient(configuration);
    provider.connectClient();

    verify(runnableRetrier).scheduleRetry(any(), any());
    assertThat(provider.getClient().isConnected(), is(false));
  }

  @Test
  public void badConnectionErroredRetrierStarted() {
    RunnableRetrier runnableRetrier = mock(RunnableRetrier.class);
    ApiPlatformClient platformClient = mock(ApiPlatformClient.class);
    overrideVariable("retrier").in(provider).with(runnableRetrier);
    overrideVariable("client").in(provider).with(platformClient);
    doThrow(new InternalError()).when(platformClient).connect();

    provider.configureClient(configuration);
    provider.connectClient();

    verify(runnableRetrier).scheduleRetry(any(), any());
    assertThat(provider.getClient().isConnected(), is(false));
  }

  private RunnableRetrier<String> retrier() {
    return new BackoffRunnableRetrierFactory(new GatewayConfiguration()).platformConnectionRetrier();
  }

  private String platformUri() {
    return format(URI, platformServer.getHttpServer().getPort());
  }

  private void mockValidConfiguration() throws IOException {
    platformServer.getHttpServer().setResponse(builder().body(serializeMe()).build());
    coreServicesServer.getHttpServer().setResponse(builder().body(serializeMe()).build());
    when(configuration.platformClient().getPlatformUri()).thenReturn(platformUri());
    when(configuration.platformClient().getClientId()).thenReturn(CLIENT);
    when(configuration.platformClient().getClientSecret()).thenReturn(CLIENT);
  }

  private String serializeMe() throws IOException {
    Organization organization = new Organization();
    organization.setId("id");

    User user = new User();
    user.setOrganization(organization);

    OrganizationClient client = new OrganizationClient();
    client.setOrganizationId("subOrgId");

    Me me = new Me();
    me.setUser(user);
    me.setClient(client);

    StringWriter stringWriter = new StringWriter();
    new ObjectMapper().writeValue(stringWriter, me);

    return stringWriter.toString();
  }

}
