/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.client;

import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.anypoint.tests.http.SimpleHttpServerResponse.builder;
import static java.lang.String.format;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertThat;
import static org.mockito.Mockito.RETURNS_DEEP_STUBS;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mule.runtime.core.api.config.MuleManifest.getProductVersion;

import org.mule.functional.util.http.HttpMessage;
import org.mule.tck.junit4.AbstractMuleTestCase;

import com.mulesoft.anypoint.tests.rules.HttpServerRule;
import com.mulesoft.mule.runtime.gw.api.config.GatewayConfiguration;

import java.io.IOException;
import java.net.InetAddress;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.util.List;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;

public class ApiPlatformClientRequestsTestCase extends AbstractMuleTestCase {

  private static final String URI = "http://localhost:%s";
  private static final String CLIENT = "client";

  @Rule
  public HttpServerRule platformServer = new HttpServerRule("platformPort");

  private ApiPlatformClient platformClient;
  private GatewayConfiguration configuration;

  @Before
  public void setUp() {
    this.platformClient = new ApiPlatformClient();
    configuration = mock(GatewayConfiguration.class, RETURNS_DEEP_STUBS);
  }

  @Test
  public void requestsContainsCustomUserAgentHeader() throws IOException, URISyntaxException {
    mockValidConfiguration();
    this.platformClient.configure(configuration);

    this.platformClient.getApi(API_KEY, "someEntityTag");

    HttpMessage request = platformServer.getHttpServer().getLastHttpRequest();
    List<String> headerKey = request.getHeaders().keySet().stream().filter("User-Agent"::equalsIgnoreCase).collect(toList());

    assertThat(headerKey.size(), is(1));
    assertThat(request.getHeaders().get(headerKey.get(0)).size(), is(1));
    assertThat(request.getHeaders().get(headerKey.get(0)).contains(desiredUserAgentHeaderValue()), is(true));
  }

  @Test
  public void apiRequestContainsVersion() throws IOException, URISyntaxException {
    mockValidConfiguration();
    this.platformClient.configure(configuration);

    this.platformClient.getApi(API_KEY, "someEntityTag");
    assertThat(platformServer.getHttpServer().getLastHttpRequest().getQueryParams().containsKey("version"), is(true));
  }

  @Test
  public void postMetricsFreesResources() throws IOException, URISyntaxException {
    mockValidConfiguration();
    this.platformClient.configure(configuration);

    // The default configuration for the client is to hold to open connections to the same path.
    // If we can execute the third request it means that we are safely handling the response.
    this.platformClient.postMetrics("{}");
    this.platformClient.postMetrics("{}");
    this.platformClient.postMetrics("{}");
  }

  @Test
  public void templateRequestContainsCorrectParams() throws IOException, URISyntaxException {
    mockValidConfiguration();
    this.platformClient.configure(configuration);

    this.platformClient.getPolicyTemplateMetadata(POLICY_TEMPLATE_KEY);
    assertThat(platformServer.getHttpServer().getLastHttpRequest().getQueryParams().containsKey("muleVersion"), is(true));
    assertThat(platformServer.getHttpServer().getLastHttpRequest().getQueryParams().get("groupId"),
               is(POLICY_TEMPLATE_KEY.getGroupId()));
    assertThat(platformServer.getHttpServer().getLastHttpRequest().getQueryParams().get("assetId"),
               is(POLICY_TEMPLATE_KEY.getAssetId()));
    assertThat(platformServer.getHttpServer().getLastHttpRequest().getQueryParams().get("version"),
               is(POLICY_TEMPLATE_KEY.getVersion()));
    assertThat(platformServer.getHttpServer().getLastHttpRequest().getQueryParams().get("source"), is("internal"));
  }

  private String platformUri() {
    return format(URI, platformServer.getHttpServer().getPort());
  }

  private void mockValidConfiguration() {
    platformServer.getHttpServer().setResponse(builder().body(apiPayload()).statusCode(200).build());
    when(configuration.platformClient().getPlatformUri()).thenReturn(platformUri());
    when(configuration.platformClient().getMetricsBaseUri()).thenReturn(platformUri());
    when(configuration.platformClient().getClientId()).thenReturn(CLIENT);
    when(configuration.platformClient().getClientSecret()).thenReturn(CLIENT);
    when(configuration.platformClient().getPolicyTemplateSource()).thenReturn("internal");
  }

  private String apiPayload() {
    return "{\"active\":false,\"apiId\":2,\"instanceName\":\"API1other\",\"productVersion\":\"\",\"groupId\":\"\",\"assetId\":\"\","
        + "\"version\":\"1.1.1\",\"environmentId\":\"5269\",\"endpointType\":\"http\",\"legacyApiIdentifier\":99,\"policies\":[],"
        + "\"endpointUri\":\"anUri\",\"tiers\":null}";
  }

  public String desiredUserAgentHeaderValue() throws UnknownHostException {
    return format("API Gateway %s (Java/%s) (%s)",
                  getProductVersion(), System.getProperty("java.version"), InetAddress.getLocalHost().getHostName());
  }

}
