/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.client;

import static com.mulesoft.anypoint.tests.ExceptionChecker.expect;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.mule.runtime.gw.client.ApiPlatformClient.RestClientStatus.INITIALISED;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideVariable;
import static org.apache.http.auth.AuthScope.ANY_REALM;
import static org.apache.http.auth.AuthScope.ANY_SCHEME;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.instanceOf;
import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.mock;

import org.mule.tck.junit4.rule.SystemProperty;

import com.mulesoft.mule.runtime.gw.api.config.GatewayConfiguration;
import com.mulesoft.mule.runtime.gw.client.dto.ApiClientDto;
import com.mulesoft.mule.runtime.gw.client.exception.EntityUnparsingException;
import com.mulesoft.mule.runtime.gw.client.exception.HttpResponseException;
import com.mulesoft.mule.runtime.gw.client.httpclient.GatewayHttpClient;
import com.mulesoft.mule.runtime.gw.client.httpclient.connection.RestartableConnectionManager;
import com.mulesoft.mule.runtime.gw.client.mocks.CloseVerifiedInputStream;
import com.mulesoft.mule.runtime.gw.client.model.ApiResponse;
import com.mulesoft.mule.runtime.gw.client.model.Me;
import com.mulesoft.mule.runtime.gw.client.response.PlatformResponse;
import com.mulesoft.mule.runtime.gw.model.ApiTrackingInfo;
import com.mulesoft.mule.runtime.gw.reflection.Inspector;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.List;
import java.util.Map;

import org.apache.http.auth.AuthScope;
import org.apache.http.auth.Credentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.Configurable;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;

public class ApiPlatformClientTestCase extends HttpClientTestCase {

  @Rule
  public SystemProperty connectTimeout = new SystemProperty("anypoint.platform.connect_timeout", "12345");

  @Rule
  public SystemProperty readTimeout = new SystemProperty("anypoint.platform.read_timeout", "67890");

  @Rule
  public SystemProperty readTimeoutTemplates = new SystemProperty("anypoint.platform.templates_read_timeout", "445566");

  @Rule
  public SystemProperty proxyHost = new SystemProperty("anypoint.platform.proxy_host", PROXY_HOST);

  @Rule
  public SystemProperty proxyPort = new SystemProperty("anypoint.platform.proxy_port", String.valueOf(PROXY_PORT));

  @Rule
  public SystemProperty proxyUsername = new SystemProperty("anypoint.platform.proxy_username", "luci");

  @Rule
  public SystemProperty proxyPassword = new SystemProperty("anypoint.platform.proxy_password", "fer");

  private static final String PROXY_HOST = "sit.com";
  private static final int PROXY_PORT = 666;

  private static final Long API_ID = 1L;
  private static final String ENVIRONMENT_ID = "envId";
  private static final String ORG_ID = "organizationId";

  private ApiPlatformClient platformClient;
  private HttpClient configuredClient;
  private HttpClient configuredUnauthenticatedClient;
  private HttpClient metricsClient;

  @Before
  public void setUp() throws URISyntaxException {
    super.setUp();
    this.platformClient = new ApiPlatformClient();
    this.platformClient.configure(new GatewayConfiguration());
    overrideVariable("orgId").in(platformClient).with(ORG_ID);
    configuredClient = new Inspector(platformClient).read("client.httpClient");
    configuredUnauthenticatedClient = new Inspector(platformClient).read("unauthenticatedClient.httpClient");
    metricsClient = new Inspector(platformClient).read("metricsClient.httpClient");
    overrideVariable("client").in(platformClient).with(new GatewayHttpClient(httpClient, connectionManager(), 120));
    overrideVariable("unauthenticatedClient").in(platformClient)
        .with(new GatewayHttpClient(httpClient, connectionManager(), 120));
    overrideVariable("metricsClient").in(platformClient)
        .with(new GatewayHttpClient(httpClient, connectionManager(), 120));
  }

  @Test
  public void authenticatedHttpClientConfiguration() {
    httpClientConfiguration(configuredClient, 67890);
  }

  @Test
  public void unauthenticatedHttpClientConfiguration() {
    httpClientConfiguration(configuredUnauthenticatedClient, 445566);
  }

  @Test
  public void metricsClientConfiguration() {
    httpClientConfiguration(metricsClient, 445566);
  }

  private void httpClientConfiguration(HttpClient client, int readTimeOut) {

    assertThat(client, instanceOf(Configurable.class));

    RequestConfig requestConfig = ((Configurable) client).getConfig();

    assertThat(requestConfig.getConnectTimeout(), is(12345));
    assertThat(requestConfig.getSocketTimeout(), is(readTimeOut));

    assertThat(requestConfig.getProxy().getHostName(), is(PROXY_HOST));
    assertThat(requestConfig.getProxy().getPort(), is(PROXY_PORT));

    CredentialsProvider credentialsProvider = new Inspector(client).read("credentialsProvider");
    AuthScope authScope = new AuthScope(PROXY_HOST, PROXY_PORT, ANY_REALM, ANY_SCHEME);
    Credentials credentials = credentialsProvider.getCredentials(authScope);
    assertThat(credentials.getUserPrincipal().getName(), is("luci"));
    assertThat(credentials.getPassword(), is("fer"));
  }

  @Test
  public void connectSetsUpOrganizationId() {
    CloseVerifiedInputStream payload = me();
    httpClientExecuteReturns(200, payload);

    Me me = platformClient.connect().entity();

    assertConsumed(payload);
    assertThat(new Inspector(platformClient).read("orgId"), is(me.getUser().getOrganization().getId()));
    assertThat(new Inspector(platformClient).read("status"), is(INITIALISED));
  }

  @Test
  public void activeEndpointConsumesEntityOnValidStatusCode() {
    CloseVerifiedInputStream payload = invalidPayload();
    httpClientExecuteReturns(201, payload);

    platformClient.activateEndpoint(ORG_ID, ENVIRONMENT_ID, API_ID);

    assertConsumed(payload);
  }

  @Test
  public void activateEndpointConsumesEntityOnInvalidStatusCode() {
    CloseVerifiedInputStream payload = invalidPayload();
    httpClientExecuteReturns(404, payload);

    expect(HttpResponseException.class, () -> platformClient.activateEndpoint(ORG_ID, ENVIRONMENT_ID, API_ID));

    assertConsumed(payload);
  }

  @Test
  public void getApiConsumesEntityOnInvalidStatusCode() {
    CloseVerifiedInputStream payload = invalidPayload();
    httpClientExecuteReturns(404, payload);

    expect(HttpResponseException.class, () -> platformClient.getApi(API_KEY, null));

    assertConsumed(payload);
  }

  @Test
  public void getApiConsumesEntityOn304StatusCode() {
    CloseVerifiedInputStream payload = emptyPayload();
    httpClientExecuteReturns(304, payload);

    platformClient.getApi(API_KEY, "someEntityTag");

    assertConsumed(payload);
  }

  @Test
  public void getApiConsumesEntityOnInvalidPayload() {
    CloseVerifiedInputStream payload = invalidPayload();
    httpClientExecuteReturns(200, payload);

    expect(EntityUnparsingException.class, () -> platformClient.getApi(API_KEY, null));

    assertConsumed(payload);
  }

  @Test
  public void getApiConsumesEntityOnValidPayload() {
    CloseVerifiedInputStream payload = getApiPayload();
    httpClientExecuteReturns(200, payload);

    PlatformResponse<ApiResponse> response = platformClient.getApi(API_KEY, "someEntityTag");

    ApiResponse apiResponse = response.entity();
    assertThat(response.statusCode(), is(200));
    assertThat(apiResponse.getTrackingInfo(), instanceOf(ApiTrackingInfo.class));
    assertThat(apiResponse.countPolicies(), is(0));
    assertThat(apiResponse.hasUpdates(), is(true));
    assertThat(apiResponse.getTrackingInfo().getInstanceName(), is("API1other"));
    assertThat(apiResponse.getTrackingInfo().getVersion(), is("1.1.1"));
    assertConsumed(payload);
  }

  @Test
  public void getApiConsumesEntityOnValidPayloadWithIdentityManagement() {
    CloseVerifiedInputStream payload = getApiPayloadWithIdentityManagement();
    httpClientExecuteReturns(200, payload);

    PlatformResponse<ApiResponse> response = platformClient.getApi(API_KEY, "someEntityTag");

    ApiResponse apiResponse = response.entity();
    assertThat(response.statusCode(), is(200));
    assertThat(apiResponse.countPolicies(), is(1));
    assertThat(getConfigDataFromApiResponse(apiResponse).size(), is(7));
    assertThat(getConfigDataFromApiResponse(apiResponse).get("identityManagementTokenUrl"),
               is("https://localhost.com:9031/as/token.oauth2"));
    assertThat(getConfigDataFromApiResponse(apiResponse).get("identityManagementClientId"), is("someId"));
    assertThat(getConfigDataFromApiResponse(apiResponse).get("identityManagementClientSecret"), is("someSecret"));
    assertConsumed(payload);
  }

  @Test
  public void getApiClientConsumesEntityOnInvalidStatusCode() {
    CloseVerifiedInputStream payload = invalidPayload();
    httpClientExecuteReturns(503, payload);

    expect(HttpResponseException.class, () -> platformClient.getApiClients(ORG_ID, ENVIRONMENT_ID, API_ID));

    assertConsumed(payload);
  }

  @Test
  public void getApiClientConsumesEntityOnInvalidPayload() {
    CloseVerifiedInputStream payload = invalidPayload();
    httpClientExecuteReturns(200, payload);

    expect(EntityUnparsingException.class, () -> platformClient.getApiClients(ORG_ID, ENVIRONMENT_ID, API_ID));

    assertConsumed(payload);
  }

  @Test
  public void getApiClientConsumesEntityOnValidPayload() {
    CloseVerifiedInputStream payload = apiClientsPayload();
    httpClientExecuteReturns(200, payload);

    PlatformResponse<List<ApiClientDto>> response = platformClient.getApiClients(ORG_ID, ENVIRONMENT_ID, API_ID);

    ApiClientDto apiClientDto = response.entity().get(0);
    assertThat(response.entity(), hasSize(1));
    assertThat(apiClientDto.getId(), is("8a3ea651-7f1c-454f-b7db-7ace47427be3"));
    assertThat(apiClientDto.getName(), is("testClient"));
    assertThat(apiClientDto.getSlaTierId().get(), is(1119331212));
    assertThat(apiClientDto.getContractId(), is("1595969484"));
    assertConsumed(payload);
  }

  @Test
  public void postMetricsEvents() throws IOException {
    // Are we going to receive something? What are the different status code we may receive?
    httpClientExecuteReturns(200, emptyPayload());

    int statusCode = platformClient.postMetrics("Hello World");
    assertThat(statusCode, is(200));
  }

  private Map<String, Object> getConfigDataFromApiResponse(ApiResponse response) {
    return response.getPolicySet().getPolicyDefinitions().get(0).getConfigurationData().getConfiguration();
  }

  private CloseVerifiedInputStream me() {
    return new CloseVerifiedInputStream(("{\"user\":{\"id\":\"john.doe\","
        + "\"username\":\"john.doe\","
        + "\"email\":\"john.doe@acme.com\","
        + "\"firstName\":\"John\","
        + "\"lastName\":\"Doe\","
        + "\"organization\":{"
        + "\"name\":\"Acme, Inc.\","
        + "\"id\":\"18763a84-2c40-4988-a75c-164f0335b7b5\","
        + "\"domain\":\"acme.com\"}},"
        + "\"client\":{\"org_id\":\"d83b3280-4ea8-4c3b-a8a2-ec6e589574f4\"}}").getBytes());
  }

  private CloseVerifiedInputStream apiClientsPayload() {
    String oneClientPayload =
        "{\"count\": 1}\n {\"clientId\":\"8a3ea651-7f1c-454f-b7db-7ace47427be3\","
            + "\"clientSecret\":\"21bcc3fa-a033-4f4d-b2b1-66832fc14ee5\","
            + "\"clientName\":\"testClient\","
            + "\"masterOrganizationId\":\"b13ef505-e284-4572-869b-9456d1790aa5\","
            + "\"slaTierId\":\"1119331212\","
            + "\"applicationId\":\"1246028032\","
            + "\"contractId\":\"1595969484\","
            + "\"redirectUris\":[\"urn:ietf:wg:oauth:2.0:oob\",\"urn:ietf:wg:oauth:2.0:oob:auto\",\"http://localhost:8080\"]}";

    return new CloseVerifiedInputStream(oneClientPayload.getBytes());
  }

  private CloseVerifiedInputStream getApiPayload() {
    String oneApiPayload =
        "{\"active\":false,\"apiId\":2,\"instanceName\":\"API1other\",\"productVersion\":\"\",\"groupId\":\"\",\"assetId\":\"\","
            + "\"version\":\"1.1.1\",\"environmentId\":\"5269\",\"endpointType\":\"http\",\"legacyApiIdentifier\":99,\"policies\":[],"
            + "\"endpointUri\":\"anUri\",\"tiers\":null}";

    return new CloseVerifiedInputStream(oneApiPayload.getBytes());
  }

  private CloseVerifiedInputStream getApiPayloadWithIdentityManagement() {
    String oneApiPayload =
        "{\"active\":false,\"apiId\":2,\"name\":\"API1other\",\"productVersion\":\"\",\"groupId\":\"\",\"assetId\":\"\","
            + "\"version\":\"1.1.1\",\"environmentId\":\"5269\",\"endpointType\":\"http\",\"legacyApiIdentifier\":99,\"policies\":[{  \n"
            +
            "         \"policyTemplateId\":\"78\",\n" +
            "         \"order\":1,\n" +
            "         \"pointcutData\":null,\n" +
            "         \"policyId\":81322,\n" +
            "         \"configuration\":{  \n" +
            "            \"scopes\":null,\n" +
            "            \"exposeHeaders\":true\n" +
            "         },\n" +
            "         \"template\":{  \n" +
            "            \"groupId\":\"68ef9520-24e9-4cf2-b2f5-620025690913\",\n" +
            "            \"assetId\":\"pingfederate-access-token-enforcement\",\n" +
            "            \"assetVersion\":\"1.2.0\"\n" +
            "         },\n" +
            "         \"version\":1509629177049\n" +
            "      }],"
            + "\"endpointUri\":\"anUri\",\"tiers\":null, \"identityManagement\":{  \n" +
            "      \"tokenUrl\":\"https://localhost.com:9031/as/token.oauth2\",\n" +
            "      \"clientId\":\"someId\",\n" +
            "      \"clientSecret\":\"someSecret\"\n" +
            "   }}";

    return new CloseVerifiedInputStream(oneApiPayload.getBytes());
  }

  private RestartableConnectionManager connectionManager() {
    return mock(RestartableConnectionManager.class);
  }
}
