/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.client;

import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideVariable;
import static org.apache.http.entity.ContentType.TEXT_PLAIN;
import static org.apache.http.protocol.HTTP.CONTENT_TYPE;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import com.mulesoft.anypoint.tests.logger.MockLogger;
import com.mulesoft.anypoint.tests.logger.TraceLine;
import com.mulesoft.mule.runtime.gw.client.httpclient.interceptors.TraceInterceptor;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.ProtocolVersion;
import org.apache.http.StatusLine;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.message.BasicHeader;
import org.apache.http.message.BasicHttpResponse;
import org.apache.http.message.BasicStatusLine;
import org.apache.http.protocol.HttpContext;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

public class TraceInterceptorTestCase {

  private static final String REQUEST_ID = "request_id";
  private static final String APPLICATION_OCTET_STREAM = "application/octet-stream";

  private static HttpContext context;
  private static HttpRequest simpleRequest;
  private static HttpResponse simpleResponse;
  private static HttpResponse complexResponse;
  private static HttpResponse complexResponseWithNullHeaderValue;

  private MockLogger logger;
  private TraceInterceptor interceptor;

  @BeforeClass
  public static void setUpClass() throws URISyntaxException {
    simpleRequest = buildSimpleRequest();
    simpleResponse = buildSimpleResponse();
    complexResponse = buildComplexResponse(applicationOctectStreamContentType());
    complexResponseWithNullHeaderValue = buildComplexResponse(nullContentType());

    context = mock(HttpContext.class);
    when(context.getAttribute(REQUEST_ID)).thenReturn(1L);
  }

  @Before
  public void setUp() {
    logger = new MockLogger();
    logger.enableTrace(true);

    interceptor = new TraceInterceptor(logger);
    overrideVariable("logger").in(interceptor).with(logger);
  }

  @Test
  public void logSimpleRequest() throws IOException, HttpException, URISyntaxException {
    interceptor.process(simpleRequest, context);
    assertThat(logger.lines().get(0), is(
                                         new TraceLine("\nREQUEST #1: POST www.testthelogger.com/testmeplease\n" +
                                             "Content-Type: application/json\n" +
                                             "Connection: Keep-Alive\n")));
  }

  @Test
  public void logResponseWithTextBody() throws IOException, HttpException {
    interceptor.process(simpleResponse, context);
    assertThat(logger.lines().get(0), is(
                                         new TraceLine("\n"
                                             + "RESPONSE #1: Status Code: 200\n"
                                             + "Content-Type: text/plain\n\n\n"
                                             + "RESPONSE BODY #1: Hello World\n")));
  }

  @Test
  public void responseBodyIsNotLoggedWhenNoTextContentType() throws IOException, HttpException {
    interceptor.process(complexResponse, context);
    assertThat(logger.lines().get(0), is(
                                         new TraceLine("\n"
                                             + "RESPONSE #1: Status Code: 200\n"
                                             + "Content-Type: application/octet-stream\n"
                                             + "\n"
                                             + "This type of payload will not be logged")));
  }

  @Test
  public void responseBodyIsNotLoggedWhenNullContentTypeValue() throws IOException, HttpException {
    interceptor.process(complexResponseWithNullHeaderValue, context);
    assertThat(logger.lines().get(0), is(
                                         new TraceLine("\n"
                                             + "RESPONSE #1: Status Code: 200\n"
                                             + "Content-Type: application/octet-stream\n"
                                             + "\n"
                                             + "This type of payload will not be logged")));
  }

  @Test
  public void requestIsNotLoggedWhenTraceIsNotEnabled() throws IOException, HttpException {
    logger.enableTrace(false);

    interceptor.process(simpleRequest, context);
    assertThat(logger.lines().size(), is(0));
  }

  @Test
  public void responseIsNotLoggedWhenTraceIsNotEnabled() throws IOException, HttpException {
    logger.enableTrace(false);

    interceptor.process(simpleResponse, context);
    assertThat(logger.lines().size(), is(0));
  }

  private static HttpResponse buildComplexResponse(Header contentTypeHeader) {
    HttpEntity mockEntity = mock(HttpEntity.class);
    when(mockEntity.getContentType()).thenReturn(contentTypeHeader);

    HttpResponse complexResponse = new BasicHttpResponse(statusLine());
    complexResponse.setStatusCode(200);
    complexResponse.setHeader(CONTENT_TYPE, APPLICATION_OCTET_STREAM);
    complexResponse.setEntity(mockEntity);

    return complexResponse;
  }

  private static HttpResponse buildSimpleResponse() {
    HttpResponse simpleResponse = new BasicHttpResponse(statusLine());
    simpleResponse.setStatusCode(200);
    simpleResponse.setHeader(CONTENT_TYPE, TEXT_PLAIN.getMimeType());
    simpleResponse.setEntity(new StringEntity("Hello World", "UTF-8"));
    return simpleResponse;
  }

  private static HttpRequest buildSimpleRequest() throws URISyntaxException {
    HttpRequest simpleRequest = new HttpPost(new URI("www.testthelogger.com/testmeplease"));
    simpleRequest.setHeader("Content-Type", "application/json");
    simpleRequest.setHeader("Connection", "Keep-Alive");
    return simpleRequest;
  }

  private static StatusLine statusLine() {
    ProtocolVersion protocolVersion = new ProtocolVersion("HTTP", 1, 1);
    return new BasicStatusLine(protocolVersion, 200, "Hello World");
  }

  private static Header applicationOctectStreamContentType() {
    return new BasicHeader("Content-Type", APPLICATION_OCTET_STREAM);
  }

  private static Header nullContentType() {
    return new BasicHeader("Content-Type", null);
  }


}
