/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.client.httpclient;

import static java.util.concurrent.TimeUnit.MILLISECONDS;

import com.mulesoft.mule.runtime.gw.client.httpclient.connection.RestartableConnectionManager;

import java.io.IOException;

import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.conn.ClientConnectionManager;
import org.apache.http.params.HttpParams;
import org.apache.http.protocol.HttpContext;

public class GatewayHttpClient implements HttpClient {

  private final HttpClient httpClient;
  private final RestartableConnectionManager connectionManager;
  private final int timeout;

  public GatewayHttpClient(HttpClient httpClient, RestartableConnectionManager connectionManager, int timeout) {
    this.httpClient = httpClient;
    this.connectionManager = connectionManager;
    this.timeout = timeout;
  }

  public void cleanConnections() {
    connectionManager.closeExpiredConnections();
    connectionManager.closeIdleConnections(timeout, MILLISECONDS);
  }

  @Override
  public HttpParams getParams() {
    return httpClient.getParams();
  }

  @Override
  public ClientConnectionManager getConnectionManager() {
    return httpClient.getConnectionManager();
  }

  @Override
  public HttpResponse execute(HttpUriRequest request) throws IOException {
    try {
      return httpClient.execute(request);
    } catch (Error | IOException e) {
      connectionManager.restart();
      throw e;
    }
  }

  @Override
  public HttpResponse execute(HttpUriRequest request, HttpContext context) throws IOException {
    try {
      return httpClient.execute(request, context);
    } catch (Error | IOException e) {
      connectionManager.restart();
      throw e;
    }
  }

  @Override
  public HttpResponse execute(HttpHost target, HttpRequest request) throws IOException {
    try {
      return httpClient.execute(target, request);
    } catch (Error | IOException e) {
      connectionManager.restart();
      throw e;
    }
  }

  @Override
  public HttpResponse execute(HttpHost target, HttpRequest request, HttpContext context)
      throws IOException {
    try {
      return httpClient.execute(target, request, context);
    } catch (Error | IOException e) {
      connectionManager.restart();
      throw e;
    }
  }

  @Override
  public <T> T execute(HttpUriRequest request, ResponseHandler<? extends T> responseHandler)
      throws IOException {
    try {
      return httpClient.execute(request, responseHandler);
    } catch (Error | IOException e) {
      connectionManager.restart();
      throw e;
    }
  }

  @Override
  public <T> T execute(HttpUriRequest request, ResponseHandler<? extends T> responseHandler, HttpContext context)
      throws IOException {
    try {
      return httpClient.execute(request, responseHandler, context);
    } catch (Error | IOException e) {
      connectionManager.restart();
      throw e;
    }
  }

  @Override
  public <T> T execute(HttpHost target, HttpRequest request, ResponseHandler<? extends T> responseHandler)
      throws IOException {
    try {
      return httpClient.execute(target, request, responseHandler);
    } catch (Error | IOException e) {
      connectionManager.restart();
      throw e;
    }
  }

  @Override
  public <T> T execute(HttpHost target, HttpRequest request, ResponseHandler<? extends T> responseHandler, HttpContext context)
      throws IOException {
    try {
      return httpClient.execute(target, request, responseHandler, context);
    } catch (Error | IOException e) {
      connectionManager.restart();
      throw e;
    }
  }

  public void shutdown() {
    this.connectionManager.shutdown();
  }
}
