/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.client.adapter;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;

import com.mulesoft.mule.runtime.gw.api.client.Client;
import com.mulesoft.mule.runtime.gw.api.client.Client.ClientBuilder;
import com.mulesoft.mule.runtime.gw.api.contract.Contract;
import com.mulesoft.mule.runtime.gw.api.contract.NoSla;
import com.mulesoft.mule.runtime.gw.api.contract.Sla;
import com.mulesoft.mule.runtime.gw.api.contract.tier.SingleTier;
import com.mulesoft.mule.runtime.gw.api.contract.tier.Tier;
import com.mulesoft.mule.runtime.gw.client.dto.ApplicationDto;
import com.mulesoft.mule.runtime.gw.client.dto.ContractDto;
import com.mulesoft.mule.runtime.gw.client.dto.LimitDto;
import com.mulesoft.mule.runtime.gw.client.dto.SlaDto;

import java.util.Arrays;
import java.util.function.Consumer;

import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

public class ContractBuilderTestCase {

  private ClientBuilder clientBuilder = Client.builder();

  @Rule
  public ExpectedException thrown = ExpectedException.none();

  @Test
  public void singleTierContractBuild() {
    Client client = client();
    ContractDto contractDto = contractDTO(client, singleSlaDto());
    LimitDto limitDto = contractDto.getTier().getLimits().get(0);

    Contract contract = new ContractBuilder().from(contractDto).build();

    assertThat(contract.client(), is(client));
    assertThat(contract.sla(), is(new Sla(1, toTier(limitDto))));
  }

  @Test
  public void multipleTiersContractBuild() {
    Client client = client();
    ContractDto contractDto = contractDTO(client, doubleSlaDto());
    LimitDto limitDto = contractDto.getTier().getLimits().get(0);
    LimitDto anotherLimitDto = contractDto.getTier().getLimits().get(1);

    Contract contract = new ContractBuilder().from(contractDto).build();

    assertThat(contract.client(), is(client));
    assertThat(contract.sla(), is(new Sla(2, toTier(limitDto), toTier(anotherLimitDto))));
  }

  @Test
  public void exceptionWhenNullClientId() {
    thrown.expectMessage("Cannot create a Contract without client id");

    ContractDto contractDto = contractDTO(client(), doubleSlaDto());
    contractDto.getApp().setClientId(null);

    new ContractBuilder().from(contractDto).build();
  }

  @Test
  public void exceptionWhenNullClientSecret() {
    thrown.expectMessage("Cannot create a Contract without client secret");

    ContractDto contractDto = contractDTO(client(), doubleSlaDto());
    contractDto.getApp().setClientSecret(null);

    new ContractBuilder().from(contractDto).build();
  }

  @Test
  public void noSLAWhenTierIsNull() {

    Consumer<ContractDto> whenSetTierNull = (contract) -> contract.setTier(null);

    assertNoSLA(whenSetTierNull);
  }

  @Test
  public void noSLAWhenLimitsAreNull() {

    Consumer<ContractDto> whenSetLimitsNull = (contract) -> contract.getTier().setLimits(null);

    assertNoSLA(whenSetLimitsNull);
  }

  @Test
  public void exceptionWhenNullSLAID() {
    thrown.expectMessage("Cannot create a Contract without an SLA ID");

    ContractDto contractDto = contractDTO(client(), doubleSlaDto());
    contractDto.getTier().setId(null);

    new ContractBuilder().from(contractDto).build();
  }

  private void assertNoSLA(Consumer<ContractDto> modifyContractClosure) {
    ContractDto contractDto = contractDTO(client(), singleSlaDto());
    modifyContractClosure.accept(contractDto);

    Contract contract = new ContractBuilder().from(contractDto).build();
    assertThat(contract.sla(), is(new NoSla()));
  }

  private Client client() {
    return clientBuilder.withId("someId").withSecret("someSecret").withName("some name").build();
  }

  private Tier toTier(LimitDto limitDto) {
    return new SingleTier(limitDto.getMaximumRequests(), limitDto.getTimePeriodInMilliseconds());
  }

  private ContractDto contractDTO(Client client, SlaDto slaDto) {
    ContractDto dto = new ContractDto();
    ApplicationDto app = application(client);

    dto.setApp(app);
    dto.setTier(slaDto);
    return dto;
  }

  private SlaDto singleSlaDto() {
    SlaDto dto = new SlaDto();
    dto.setId(1);
    dto.setLimits(Arrays.asList(limitDTO(10, 60000)));
    return dto;
  }

  private SlaDto doubleSlaDto() {
    SlaDto dto = new SlaDto();
    dto.setId(2);
    dto.setLimits(Arrays.asList(limitDTO(10, 60000), limitDTO(15, 4234)));
    return dto;
  }

  private LimitDto limitDTO(int maxRequests, int periodMillis) {
    LimitDto dto = new LimitDto();
    dto.setMaximumRequests(maxRequests);
    dto.setTimePeriodInMilliseconds(periodMillis);
    return dto;
  }

  private ApplicationDto application(Client client) {
    ApplicationDto dto = new ApplicationDto();
    dto.setClientId(client.id());
    dto.setClientSecret(client.secret());
    dto.setName(client.name());
    return dto;
  }
}
