/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.client.adapter;

import static com.google.common.collect.Lists.newArrayList;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static junit.framework.TestCase.assertTrue;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasItems;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static org.junit.rules.ExpectedException.none;

import org.mule.tck.junit4.AbstractMuleTestCase;

import com.mulesoft.mule.runtime.gw.api.policy.HttpResourcePointcut;
import com.mulesoft.mule.runtime.gw.api.policy.PolicyTemplateKey;
import com.mulesoft.mule.runtime.gw.client.dto.IdentityManagementDto;
import com.mulesoft.mule.runtime.gw.client.dto.PointcutDataDto;
import com.mulesoft.mule.runtime.gw.client.dto.PolicyDefinitionDto;
import com.mulesoft.mule.runtime.gw.client.dto.PolicyTemplateKeyDto;
import com.mulesoft.mule.runtime.gw.client.exception.IncompatibleApiException;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

public class PolicyDefinitionBuilderTestCase extends AbstractMuleTestCase {

  @Rule
  public ExpectedException expectedException = none();

  private PolicyDefinitionBuilder builder = new PolicyDefinitionBuilder();

  @Test
  public void withResourcePointcut() {
    Map<String, Object> configData = new HashMap<>();
    PolicyDefinitionDto dto = policyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, configData,
                                               newArrayList(pointcutData("uri", "GET"), pointcutData("anotherUri", "POST")));

    PolicyDefinition policyDefinition = builder.from(API_KEY, dto, "wsdl", null);

    assertThat(policyDefinition.getHttpResourcePointcuts(), hasSize(2));
    assertThat(policyDefinition.getHttpResourcePointcuts(), hasItems(new HttpResourcePointcut("uri", "GET"),
                                                                     new HttpResourcePointcut("anotherUri", "POST")));
    assertThat(policyDefinition.getConfigurationData().getConfiguration().size(), is(2));
    assertTrue(policyDefinition.getConfigurationData().getConfiguration().containsKey("policyId"));
    assertThat(policyDefinition.getConfigurationData().getConfiguration().get("policyId"), is(POLICY_ID));
  }

  @Test
  public void withIdentityManagementWithClient() {
    Map<String, Object> configData = new HashMap<>();
    IdentityManagementDto identityManagement = new IdentityManagementDto();
    identityManagement.setTokenUrl("127.0.0.1");
    identityManagement.setClientId("clientId");
    identityManagement.setClientSecret("clientSecret");
    PolicyDefinitionDto dto = policyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, configData, null);

    PolicyDefinition policyDefinition = builder.from(API_KEY, dto, "wsdl", identityManagement);

    assertThat(policyDefinition.getConfigurationData().getConfiguration().size(), is(5));
    assertTrue(policyDefinition.getConfigurationData().getConfiguration().containsKey("identityManagementTokenUrl"));
    assertTrue(policyDefinition.getConfigurationData().getConfiguration().containsKey("identityManagementClientId"));
    assertTrue(policyDefinition.getConfigurationData().getConfiguration().containsKey("identityManagementClientSecret"));
  }

  @Test
  public void withIdentityManagement() {
    Map<String, Object> configData = new HashMap<>();
    IdentityManagementDto identityManagement = new IdentityManagementDto();
    identityManagement.setTokenUrl("127.0.0.1");
    PolicyDefinitionDto dto = policyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, configData, null);

    PolicyDefinition policyDefinition = builder.from(API_KEY, dto, "wsdl", identityManagement);

    assertThat(policyDefinition.getConfigurationData().getConfiguration().size(), is(3));
    assertTrue(policyDefinition.getConfigurationData().getConfiguration().containsKey("identityManagementTokenUrl"));
  }

  @Test
  public void isWsdlEndpointPropertySetting() {
    Map<String, Object> configData = new HashMap<>();
    PolicyDefinitionDto dto = policyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, configData,
                                               newArrayList(pointcutData("uri", "GET"), pointcutData("anotherUri", "POST")));

    PolicyDefinition policyDefinition = builder.from(API_KEY, dto, "wsdl", null);
    assertThat(policyDefinition.getConfigurationData().getConfiguration().size(), is(2));
    assertTrue(policyDefinition.getConfigurationData().getConfiguration().containsKey("isWsdlEndpoint"));
    assertThat(policyDefinition.getConfigurationData().getConfiguration().get("isWsdlEndpoint"), is(true));

    PolicyDefinitionDto dto2 = policyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, configData,
                                                newArrayList(pointcutData("uri", "GET"), pointcutData("anotherUri", "POST")));

    PolicyDefinition policyDefinition2 = builder.from(API_KEY, dto2, "raml", null);
    assertThat(policyDefinition.getConfigurationData().getConfiguration().size(), is(2));
    assertTrue(policyDefinition2.getConfigurationData().getConfiguration().containsKey("isWsdlEndpoint"));
    assertThat(policyDefinition2.getConfigurationData().getConfiguration().get("isWsdlEndpoint"), is(false));

    PolicyDefinitionDto dto3 = policyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, configData,
                                                newArrayList(pointcutData("uri", "GET"), pointcutData("anotherUri", "POST")));

    PolicyDefinition policyDefinition3 = builder.from(API_KEY, dto3, "http", null);
    assertThat(policyDefinition.getConfigurationData().getConfiguration().size(), is(2));
    assertTrue(policyDefinition3.getConfigurationData().getConfiguration().containsKey("isWsdlEndpoint"));
    assertThat(policyDefinition3.getConfigurationData().getConfiguration().get("isWsdlEndpoint"), is(false));
  }

  @Test
  public void policyWithoutGAVisIncompatible() {
    PolicyTemplateKey templateKey = new PolicyTemplateKey(null, null, null);
    PolicyDefinitionDto dto = policyDefinition(POLICY_ID, templateKey, emptyMap(), emptyList());

    expectedException.expect(IncompatibleApiException.class);
    expectedException.expectMessage("The API is not Mule 4 compatible.");

    builder.from(API_KEY, dto, "http", null);
  }

  private PolicyDefinitionDto policyDefinition(String id, PolicyTemplateKey templateKey, Map<String, Object> configData,
                                               List<PointcutDataDto> resourcePointcuts) {
    PolicyDefinitionDto policyDefinition = new PolicyDefinitionDto();
    policyDefinition.setId(id);
    policyDefinition.setPolicyTemplateKey(new PolicyTemplateKeyDto(templateKey.getGroupId(), templateKey.getAssetId(),
                                                                   templateKey.getVersion()));
    policyDefinition.setOrder(1);
    policyDefinition.setConfiguration(configData);
    policyDefinition.setPointcutData(resourcePointcuts);
    return policyDefinition;
  }

  private PointcutDataDto pointcutData(String uri, String method) {
    PointcutDataDto pointcutData = new PointcutDataDto();
    pointcutData.setMethodRegex(method);
    pointcutData.setUriTemplateRegex(uri);
    return pointcutData;
  }
}
