/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.client.httpclient.interceptors;

import static com.mulesoft.mule.runtime.gw.api.time.DateTime.now;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.nullValue;

import org.mule.tck.probe.PollingProber;

import com.mulesoft.anypoint.tests.DescriptiveProbe;
import com.mulesoft.mule.runtime.gw.api.time.DateTime;

import org.junit.Test;

public class AccessTokensManagerTestCase {

  private static final String CCS_TOKEN = "aSuperSecureCCSToken";

  // {"typ": "JWT","alg": "HS256"}.{"iat": 1530645663,"exp": 1530645785}
  private static final String PLATFORM_TOKEN =
      "eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9.ew0KICAiaWF0IjogMTUzMDY0NTY2MywNCiAgImV4cCI6IDE1MzA2NDU3ODUNCn0=";

  // {"typ": "JWT","alg": "HS256"}.{"iat": 1530645663,"exp": 1530645673}
  private static final String SHORT_LIVED_PLATFORM_TOKEN =
      "eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9.eyJpYXQiOiAxNTMwNjQ1NjYzLCJleHAiOiAxNTMwNjQ1NjczfQ==";

  private static final String INVALID_PLATFORM_TOKEN =
      "eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9.foobar";

  private static final Long EXPIRATION_DELAY = 2L;
  private static final int EXPIRATION_DELAY_IN_MILLIS = (int) (EXPIRATION_DELAY * 1000);
  private static final long POLLING_DELAY = 200L;

  private AccessTokenManager tokenManager = new AccessTokenManager();
  private int timesCalledAccessToken = 0;
  private int timesCalledPlatformToken = 0;

  @Test
  public void tokenManagerObtainsAccessTokens() {
    DateTime instantBeforeObtainingToken = now().plusMillis(-1); // To ensure that the time is strictly before the token obtaining
                                                                 // time
    AccessTokens tokens = tokenManager.getUpdatedTokens(this::getAccessToken, this::getPlatformToken);
    DateTime instantAfterObtainingToken = now().plusMillis(1); // To ensure that the time is strictly after the token obtaining
                                                               // time

    assertThat(tokens.getCoreServicesToken(), is(CCS_TOKEN));
    assertThat(tokens.getPlatformToken(), is(PLATFORM_TOKEN));

    assertThat(instantBeforeObtainingToken.plusMillis(EXPIRATION_DELAY_IN_MILLIS).isBefore(tokens.getExpirationTime()), is(true));
    assertThat(instantAfterObtainingToken.plusMillis(EXPIRATION_DELAY_IN_MILLIS).isAfter(tokens.getExpirationTime()), is(true));
  }

  @Test
  public void tokenManagerObtainsAccessTokensDoesNotSearchTwoTimes() {
    AccessTokens tokens = tokenManager.getUpdatedTokens(this::getAccessToken, this::getPlatformToken);
    assertThat(tokens.getCoreServicesToken(), is(CCS_TOKEN));
    assertThat(tokens.getPlatformToken(), is(PLATFORM_TOKEN));
    DateTime expirationTime = tokens.getExpirationTime();

    assertThat(timesCalledAccessToken, is(1));
    assertThat(timesCalledPlatformToken, is(1));

    tokens = tokenManager.getUpdatedTokens(this::getAccessToken, this::getPlatformToken);
    assertThat(tokens.getCoreServicesToken(), is(CCS_TOKEN));
    assertThat(tokens.getPlatformToken(), is(PLATFORM_TOKEN));
    assertThat(tokens.getExpirationTime(), is(expirationTime));

    assertThat(timesCalledAccessToken, is(1));
    assertThat(timesCalledPlatformToken, is(1));
  }

  @Test
  public void shortLivedPlatformTokenExpiresAfterFirstCall() {
    AccessTokens tokens = tokenManager.getUpdatedTokens(this::getAccessToken, this::getShortLivedPlatformToken);
    assertThat(tokens.getCoreServicesToken(), is(CCS_TOKEN));
    assertThat(tokens.getPlatformToken(), is(SHORT_LIVED_PLATFORM_TOKEN));

    assertThat(timesCalledAccessToken, is(1));
    assertThat(timesCalledPlatformToken, is(1));

    tokens = tokenManager.getUpdatedTokens(this::getAccessToken, this::getPlatformToken);
    assertThat(tokens.getCoreServicesToken(), is(CCS_TOKEN));
    assertThat(tokens.getPlatformToken(), is(PLATFORM_TOKEN));

    assertThat(timesCalledAccessToken, is(2));
    assertThat(timesCalledPlatformToken, is(2));
  }

  @Test
  public void invalidPlatformTokenFailsExtractingExpirationInfo() {
    AccessTokens tokens = tokenManager.getUpdatedTokens(this::getAccessToken, this::getInvalidPlatformToken);
    assertThat(tokens.getCoreServicesToken(), is(CCS_TOKEN));
    assertThat(tokens.getPlatformToken(), is(INVALID_PLATFORM_TOKEN));
    assertThat(tokens.getExpirationTime(), nullValue());

    assertThat(timesCalledAccessToken, is(1));
    assertThat(timesCalledPlatformToken, is(1));
  }

  @Test
  public void invalidPlatformTokenFailsDoesNotTriggerRenovation() {
    AccessTokens tokens = tokenManager.getUpdatedTokens(this::getAccessToken, this::getInvalidPlatformToken);
    assertThat(tokens.getCoreServicesToken(), is(CCS_TOKEN));
    assertThat(tokens.getPlatformToken(), is(INVALID_PLATFORM_TOKEN));
    assertThat(tokens.getExpirationTime(), nullValue());

    assertThat(timesCalledAccessToken, is(1));
    assertThat(timesCalledPlatformToken, is(1));

    tokens = tokenManager.getUpdatedTokens(this::getAccessToken, this::getPlatformToken);
    assertThat(tokens.getCoreServicesToken(), is(CCS_TOKEN));
    assertThat(tokens.getPlatformToken(), is(INVALID_PLATFORM_TOKEN));
    assertThat(tokens.getExpirationTime(), nullValue());

    assertThat(timesCalledAccessToken, is(1));
    assertThat(timesCalledPlatformToken, is(1));
  }

  @Test
  public void nullPlatformTokenRetriesOnlyPlatform() {
    AccessTokens tokens = tokenManager.getUpdatedTokens(this::getAccessToken, this::getNullPlatformToken);
    assertThat(tokens.getCoreServicesToken(), is(CCS_TOKEN));
    assertThat(tokens.getPlatformToken(), nullValue());

    assertThat(timesCalledAccessToken, is(1));
    assertThat(timesCalledPlatformToken, is(1));

    tokens = tokenManager.getUpdatedTokens(this::getAccessToken, this::getPlatformToken);
    assertThat(tokens.getCoreServicesToken(), is(CCS_TOKEN));
    assertThat(tokens.getPlatformToken(), is(PLATFORM_TOKEN));

    assertThat(timesCalledAccessToken, is(1));
    assertThat(timesCalledPlatformToken, is(2));
  }

  @Test
  public void nullCoreServicesTokenRetries() {
    AccessTokens tokens = tokenManager.getUpdatedTokens(this::getNullAccessToken, this::getInvalidPlatformToken);
    assertThat(tokens.getCoreServicesToken(), nullValue());
    assertThat(tokens.getPlatformToken(), is(INVALID_PLATFORM_TOKEN));

    assertThat(timesCalledAccessToken, is(1));
    assertThat(timesCalledPlatformToken, is(1));

    tokens = tokenManager.getUpdatedTokens(this::getAccessToken, this::getPlatformToken);
    assertThat(tokens.getCoreServicesToken(), is(CCS_TOKEN));
    assertThat(tokens.getPlatformToken(), is(PLATFORM_TOKEN));

    assertThat(timesCalledAccessToken, is(2));
    assertThat(timesCalledPlatformToken, is(2));
  }


  @Test
  public void tokenManagerObtainsAccessTokensRefreshedBeforeTokenExpires() {
    AccessTokens tokens = tokenManager.getUpdatedTokens(this::getAccessToken, this::getPlatformToken);
    assertThat(timesCalledAccessToken, is(1));
    assertThat(timesCalledPlatformToken, is(1));
    assertThat(tokens.getCoreServicesToken(), is(CCS_TOKEN));
    assertThat(tokens.getPlatformToken(), is(PLATFORM_TOKEN));

    probe(() -> {
      tokenManager.getUpdatedTokens(this::getAccessToken, this::getPlatformToken);
      assertThat(timesCalledAccessToken, is(2));
      assertThat(timesCalledPlatformToken, is(2));
    });

    tokens = tokenManager.getUpdatedTokens(this::getAccessToken, this::getPlatformToken);
    assertThat(tokens.getCoreServicesToken(), is(CCS_TOKEN));
    assertThat(tokens.getPlatformToken(), is(PLATFORM_TOKEN));
  }

  protected void probe(Runnable runnable) {
    new PollingProber(EXPIRATION_DELAY_IN_MILLIS, POLLING_DELAY).check(new DescriptiveProbe(runnable));
  }

  private String getNullPlatformToken(String accessToken) {
    timesCalledPlatformToken++;
    return null;
  }

  private String getInvalidPlatformToken(String accessToken) {
    timesCalledPlatformToken++;
    return INVALID_PLATFORM_TOKEN;
  }

  private String getShortLivedPlatformToken(String accessToken) {
    timesCalledPlatformToken++;
    return SHORT_LIVED_PLATFORM_TOKEN;
  }

  private String getPlatformToken(String accessToken) {
    timesCalledPlatformToken++;
    return PLATFORM_TOKEN;
  }

  private String getNullAccessToken() {
    timesCalledAccessToken++;
    return null;
  }

  private String getAccessToken() {
    timesCalledAccessToken++;
    return CCS_TOKEN;
  }

}
