/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.client.httpclient.interceptors;

import static com.mulesoft.anypoint.tests.ExceptionChecker.expect;
import static java.util.stream.IntStream.range;
import static org.mockito.Mockito.mock;

import org.mule.runtime.core.api.util.func.CheckedConsumer;

import com.mulesoft.mule.runtime.gw.client.HttpClientTestCase;
import com.mulesoft.mule.runtime.gw.client.exception.HttpResponseException;
import com.mulesoft.mule.runtime.gw.client.exception.NotFoundException;
import com.mulesoft.mule.runtime.gw.client.mocks.CloseVerifiedInputStream;

import java.io.IOException;
import java.net.URISyntaxException;

import org.apache.http.HttpException;
import org.apache.http.protocol.HttpContext;
import org.junit.Before;
import org.junit.Test;

public class HttpResponseStatusInterceptorTestCase extends HttpClientTestCase {

  private HttpContext httpContext;
  private HttpResponseStatusInterceptor interceptor;

  @Before
  public void setUp() throws URISyntaxException {
    super.setUp();
    this.httpContext = mock(HttpContext.class);
    this.interceptor = new HttpResponseStatusInterceptor();
  }

  @Test
  public void entityConsumedOn404() throws Exception {
    CloseVerifiedInputStream payload = payload();

    expect(NotFoundException.class,
           () -> interceptor.process(response(404, payload), httpContext));

    assertConsumed(payload);
  }

  @Test
  public void entityConsumeOn1xxAnd4xxAnd5xx() {
    CheckedConsumer<Integer> checkEntityConsumed = (statusCode) -> {
      if (statusCode != 404 && statusCode != 401 && statusCode != 407) {
        String loggingPrefix = "(statusCode: " + statusCode + ")";

        CloseVerifiedInputStream payload = payload();

        expect(loggingPrefix, HttpResponseException.class,
               () -> interceptor.process(response(statusCode, payload), httpContext));

        assertConsumed(payload, loggingPrefix);
      }
    };
    executeRange(100, 200, checkEntityConsumed);
    executeRange(400, 600, checkEntityConsumed);
  }

  @Test
  public void entityNotConsumedOn401() throws IOException, HttpException {
    CloseVerifiedInputStream payload = payload();

    interceptor.process(response(401, payload), httpContext);

    assertNotConsumed(payload, "(statusCode: " + 401 + ")");
  }

  @Test
  public void entityNotConsumedFrom200To400() {
    executeRange(200, 400, (statusCode) -> {
      CloseVerifiedInputStream payload = payload();

      interceptor.process(response(statusCode, payload), httpContext);

      assertNotConsumed(payload, "(statusCode: " + statusCode + ")");
    });
  }

  private void executeRange(Integer from, Integer to, CheckedConsumer<Integer> closure) {
    range(from, to).forEach(closure::accept);
  }

  private CloseVerifiedInputStream payload() {
    return new CloseVerifiedInputStream("this is some payload".getBytes());
  }
}
