/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.client.httpclient.interceptors;

import static com.mulesoft.anypoint.tests.ExceptionChecker.expect;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.when;

import org.mule.runtime.core.api.util.func.CheckedRunnable;
import org.mule.runtime.core.api.util.func.CheckedSupplier;

import com.mulesoft.mule.runtime.gw.client.HttpClientTestCase;
import com.mulesoft.mule.runtime.gw.client.exception.EntityUnparsingException;
import com.mulesoft.mule.runtime.gw.client.exception.HttpConnectionException;
import com.mulesoft.mule.runtime.gw.client.exception.HttpResponseException;
import com.mulesoft.mule.runtime.gw.client.exception.UnauthorizedException;
import com.mulesoft.mule.runtime.gw.client.mocks.CloseVerifiedInputStream;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;

import org.junit.Before;
import org.junit.Test;

public class Oauth2ClientCredentialsInterceptorTestCase extends HttpClientTestCase {

  private static final String CCS_TOKEN = "aSuperSecureCCSToken";
  private static final String PLATFORM_TOKEN = "aSuperSecurePlatformToken";

  private URI tokenUrl;
  private URI loginUrl;
  private CloseVerifiedInputStream invalidPayload;
  private CloseVerifiedInputStream ccsPayload;
  private CloseVerifiedInputStream platformPayload;

  @Before
  public void setUp() throws URISyntaxException {
    super.setUp();
    this.tokenUrl = URI.create("http://token-url.com");
    this.loginUrl = URI.create("http://login-url.com");
    this.invalidPayload = invalidPayload();
    this.ccsPayload = validCCSPayload();
    this.platformPayload = validPlatformPayload();
  }

  @Test
  public void authenticateConsumesEntityOnInvalidStatusCode() throws IOException {
    entityConsumedOnInvalidStatusCode(oauth2()::authenticate);
  }

  @Test
  public void authenticateConsumesEntityOnInvalidPayload() throws IOException {
    entityConsumedOnInvalidPayload(oauth2()::authenticate);
  }

  @Test
  public void authenticateConsumesEntityOnValidPayload() throws IOException {
    entityConsumedOnValidPayload(oauth2()::authenticate, ccsPayload, CCS_TOKEN);
  }

  @Test
  public void authenticate503OnIOException() throws IOException {
    clientThrowsIOException();
    expect503(oauth2()::authenticate);
  }

  @Test
  public void exchangeTokenConsumesEntityOnInvalidStatusCode() throws IOException {
    entityConsumedOnInvalidStatusCode(() -> oauth2().exchangeToken(CCS_TOKEN));
  }

  @Test
  public void exchangeTokenConsumesEntityOnInvalidPayload() throws IOException {
    entityConsumedOnInvalidPayload(() -> oauth2().exchangeToken(CCS_TOKEN));
  }

  @Test
  public void exchangeTokenConsumesEntityOnValidPayload() throws IOException {
    entityConsumedOnValidPayload(() -> oauth2().exchangeToken(CCS_TOKEN), platformPayload, PLATFORM_TOKEN);
  }

  @Test
  public void exchangeToken503OnIOException() throws IOException {
    clientThrowsIOException();
    expect503(() -> oauth2().exchangeToken(CCS_TOKEN));
  }

  private void clientThrowsIOException() throws IOException {
    when(httpClient.execute(any())).thenThrow(new IOException());
  }

  private void expect503(CheckedRunnable closure) {
    expect(HttpConnectionException.class,
           closure,
           exception -> assertThat(exception.statusCode(), is(503)));
  }

  private void entityConsumedOnValidPayload(CheckedSupplier<String> tokenRetriever,
                                            CloseVerifiedInputStream payload,
                                            String expectedToken)
      throws IOException {
    httpClientExecuteReturns(200, payload);

    String token = tokenRetriever.get();

    assertThat(token, is(expectedToken));
    assertConsumed(payload);
  }

  /**
   * It is important that we test with 401, as {@link HttpResponseStatusInterceptor} will not consume the payload on this status
   * code. If you little gatewaynian are ever to change this status code, please add a test checking that 401's payload is
   * consumed.
   */
  private void entityConsumedOnInvalidStatusCode(CheckedRunnable closure) throws IOException {
    httpClientExecuteReturns(401, invalidPayload);

    expect(UnauthorizedException.class, closure);

    assertConsumed(invalidPayload);
  }

  private void entityConsumedOnInvalidPayload(CheckedRunnable closure) throws IOException {
    httpClientExecuteReturns(200, invalidPayload);

    expect(EntityUnparsingException.class, closure);

    assertConsumed(invalidPayload);
  }

  private Oauth2ClientCredentialsInterceptor oauth2() {
    return new Oauth2ClientCredentialsInterceptor(httpClient, tokenUrl, loginUrl, "clientId", "clientSecret");
  }


  private CloseVerifiedInputStream validPlatformPayload() {
    return new CloseVerifiedInputStream(("{ \"token\": \"" + PLATFORM_TOKEN + "\" }").getBytes());
  }

  private CloseVerifiedInputStream validCCSPayload() {
    return new CloseVerifiedInputStream(("{\"access_token\": \"" + CCS_TOKEN + "\",\"token_type\": \"bearer\"}\n").getBytes());
  }
}
