/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.client.session;

import static com.mulesoft.mule.runtime.gw.api.config.PlatformClientConfiguration.BACKOFF;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideLogger;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideVariable;
import static java.lang.System.clearProperty;
import static java.lang.System.setProperty;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;

import com.mulesoft.anypoint.tests.logger.DebugLine;
import com.mulesoft.anypoint.tests.logger.MockLogger;
import com.mulesoft.mule.runtime.gw.api.config.GatewayConfiguration;
import com.mulesoft.anypoint.backoff.configuration.BackoffConfiguration;
import com.mulesoft.anypoint.backoff.session.BackoffBarrier;
import com.mulesoft.anypoint.backoff.session.PlatformBackoffBarrier;
import com.mulesoft.anypoint.backoff.session.SessionMetadata;
import com.mulesoft.mule.runtime.gw.client.session.metadata.ApiPlatformSessionMetadataBuilder;

import org.junit.Before;
import org.junit.Test;

public class ApiPlatformSessionMetadataBuilderTestCase {

  private MockLogger logger;
  private ApiPlatformSessionMetadataBuilder metadataBuilder;

  @Before
  public void setUp() {
    logger = new MockLogger();
    metadataBuilder = new ApiPlatformSessionMetadataBuilder();
  }

  @Test
  public void buildEmpty() {
    BackoffBarrier backoffBarrier = backoffBarrier();

    assertThat(backoffBarrier.shouldBackoff(metadataBuilder.build(), defaultConfiguration()), is(false));
  }

  @Test
  public void requestsAndNoResponses() {
    BackoffBarrier backoffBarrier = backoffBarrier();

    SessionMetadata metadata = metadataBuilder.build();

    assertThat(backoffBarrier.shouldBackoff(metadata, defaultConfiguration()), is(false));
  }

  @Test
  public void useConfiguredSingleStatusCodesToValidate() {
    BackoffBarrier backoffBarrier = backoffBarrier();

    SessionMetadata metadata = metadataBuilder.response(666).build();

    assertThat(backoffBarrier.shouldBackoff(metadata, configurationWith(333)), is(false));
    assertThat(backoffBarrier.shouldBackoff(metadata, configurationWith(666)), is(true));
  }

  @Test
  public void useConfiguredStatusCodeListToValidate() {
    BackoffBarrier backoffBarrier = backoffBarrier();

    SessionMetadata metadata = metadataBuilder.response(666).build();

    assertThat(backoffBarrier.shouldBackoff(metadata, configurationWith(333, 666)), is(true));
    assertThat(backoffBarrier.shouldBackoff(metadata, configurationWith(333, 444)), is(false));
  }

  @Test
  public void cornerCaseFailurePercentages() {
    BackoffBarrier backoffBarrier = backoffBarrier();
    overrideLogger().in(backoffBarrier).with(logger);

    SessionMetadata metadata = metadataBuilder // 7 "666" and the rest different.
        .response(666).response(666).response(911).response(666).response(666)
        .response(666).response(314).response(666).response(271).response(666)
        .build();

    assertThat(backoffBarrier.shouldBackoff(metadata, configurationWithPercentage(70.0, 666)), is(true));
    assertThat(backoffBarrier.shouldBackoff(metadata, configurationWithPercentage(69.99, 666)), is(true));
    assertThat(backoffBarrier.shouldBackoff(metadata, configurationWithPercentage(70.01, 666)), is(false));
    assertThat(logger.lines(), hasSize(3));
    assertThat(logger.lines().get(0),
               is(new DebugLine("Request failures {}% ({}/{}) {} threshold of {}%", 70.0, 7, 10, "above or equal to",
                                70.0)));
    assertThat(logger.lines().get(1),
               is(new DebugLine("Request failures {}% ({}/{}) {} threshold of {}%", 70.0, 7, 10, "above or equal to",
                                69.99)));
    assertThat(logger.lines().get(2),
               is(new DebugLine("Request failures {}% ({}/{}) {} threshold of {}%", 70.0, 7, 10, "below", 70.01)));
  }

  @Test
  public void neverFailsWhenBackoffDisabled() {
    setProperty(BACKOFF, "false");
    BackoffBarrier backoffBarrier = backoffBarrier();

    SessionMetadata metadata = metadataBuilder
        .response(666).response(111)
        .build();

    assertThat(backoffBarrier.shouldBackoff(metadata, configurationWithPercentage(1.0, 666)), is(false));
    assertThat(backoffBarrier.shouldBackoff(metadata, configurationWithPercentage(1.0, 111)), is(false));
    assertThat(backoffBarrier.shouldBackoff(metadata, configurationWithPercentage(1.0, 707, 231)), is(false));
    clearProperty(BACKOFF);
  }

  private BackoffBarrier backoffBarrier() {
    return new PlatformBackoffBarrier();
  }

  private BackoffConfiguration configurationWith(Integer... statusCodes) {
    return configurationWithPercentage(100.0, statusCodes);
  }

  private BackoffConfiguration configurationWithPercentage(double failurePercentage, Integer... statusCodes) {
    GatewayConfiguration gatewayConfiguration = new GatewayConfiguration();
    return new BackoffConfiguration.Builder(gatewayConfiguration.platformClient().backoffEnabled())
        .statusCodes(statusCodes)
        .failurePercentage(failurePercentage).build();
  }

  private BackoffConfiguration defaultConfiguration() {
    GatewayConfiguration gatewayConfiguration = new GatewayConfiguration();
    return new BackoffConfiguration.Builder(gatewayConfiguration.platformClient().backoffEnabled())
        .statusCodes(gatewayConfiguration.platformClient().getOutagesStatusCodes())
        .build();
  }
}
