/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.client.uri;

import static com.mulesoft.anypoint.tests.ExceptionChecker.expect;

import com.mulesoft.mule.runtime.gw.client.exception.UriBuilderException;

import java.util.ArrayList;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;

import org.junit.Test;

public class HttpClientUriBuilderTestCase {

  @Test
  public void wellFormedExceptionShouldCreateAValidURI() {
    ArrayList<String> validUris = new ArrayList<String>() {

      {
        add("http://www.mulesoft.com");
        add("http://www.mulesoft.com/");
        add("http://www.mulesoft.com/a");
        add("http://www.mulesoft.com/a/b");
        add("http://www.mulesoft.com/a/b/");
        add("http://www.mulesoft.com/a/b?c=d");
        add("http://www.mulesoft.com/a/b?c=d&e=f");
      }
    };

    validUris.forEach(uri -> assertThat(new HttpClientUriBuilder().fromUri(uri).build().toString(), equalTo(uri)));
  }

  @Test
  public void pathShouldAppendThePathCorrectly() {
    assertThat(
               new HttpClientUriBuilder().fromUri("http://www.mulesoft.com").appendPath("/somePath").build().toString(),
               equalTo("http://www.mulesoft.com/somePath"));

    assertThat(
               new HttpClientUriBuilder().fromUri("http://www.mulesoft.com/").appendPath("/somePath/").build().toString(),
               equalTo("http://www.mulesoft.com/somePath/"));

    assertThat(
               new HttpClientUriBuilder().fromUri("http://www.mulesoft.com/a").appendPath("/somePath/someOtherPathMember").build()
                   .toString(),
               equalTo("http://www.mulesoft.com/a/somePath/someOtherPathMember"));

    assertThat(
               new HttpClientUriBuilder().fromUri("http://www.mulesoft.com/a/b").appendPath("/").build().toString(),
               equalTo("http://www.mulesoft.com/a/b/"));

    assertThat(
               new HttpClientUriBuilder().fromUri("http://www.mulesoft.com/a/b/").appendPath("/somePath").build().toString(),
               equalTo("http://www.mulesoft.com/a/b/somePath"));

    assertThat(
               new HttpClientUriBuilder().fromUri("http://www.mulesoft.com/a/b?c=d").appendPath("/somePath").build().toString(),
               equalTo("http://www.mulesoft.com/a/b/somePath?c=d"));

    assertThat(
               new HttpClientUriBuilder().fromUri("http://www.mulesoft.com/a/b?c=d&e=f").appendPath("/somePath").build()
                   .toString(),
               equalTo("http://www.mulesoft.com/a/b/somePath?c=d&e=f"));

    assertThat(
               new HttpClientUriBuilder().fromUri("http://www.mulesoft.com/a/b?c=d&e=f").appendPath("/somePath")
                   .queryParam("y", "z")
                   .build().toString(),
               equalTo("http://www.mulesoft.com/a/b/somePath?y=z"));
  }

  @Test
  public void queryParamShouldSetTheParamsCorrectly() {
    assertThat(
               new HttpClientUriBuilder().fromUri("http://www.mulesoft.com").queryParam("a", "b").build().toString(),
               equalTo("http://www.mulesoft.com/?a=b"));

    assertThat(
               new HttpClientUriBuilder().fromUri("http://www.mulesoft.com/").queryParam("a", "b").build().toString(),
               equalTo("http://www.mulesoft.com/?a=b"));

    assertThat(
               new HttpClientUriBuilder().fromUri("http://www.mulesoft.com/a").queryParam("a", "b").build().toString(),
               equalTo("http://www.mulesoft.com/a?a=b"));

    assertThat(
               new HttpClientUriBuilder().fromUri("http://www.mulesoft.com/a/b").queryParam("a", "b").build().toString(),
               equalTo("http://www.mulesoft.com/a/b?a=b"));

    assertThat(
               new HttpClientUriBuilder().fromUri("http://www.mulesoft.com/a/b/").queryParam("a", "b").build().toString(),
               equalTo("http://www.mulesoft.com/a/b/?a=b"));

    assertThat(
               new HttpClientUriBuilder().fromUri("http://www.mulesoft.com/a/b?c=d").queryParam("a", "b").build().toString(),
               equalTo("http://www.mulesoft.com/a/b?a=b"));

    assertThat(
               new HttpClientUriBuilder().fromUri("http://www.mulesoft.com/a/b?c=d&e=f").queryParam("a", "b").build().toString(),
               equalTo("http://www.mulesoft.com/a/b?a=b"));

    assertThat(
               new HttpClientUriBuilder().fromUri("http://www.mulesoft.com/a/b?c=d&e=f").queryParam("a", "b").queryParam("y", "z")
                   .build().toString(),
               equalTo("http://www.mulesoft.com/a/b?a=b&y=z"));
  }

  @Test
  public void paramsShouldBeReplacedCorrectly() {
    assertThat(
               new HttpClientUriBuilder().fromUri("http://www.mulesoft.com").appendPath("/{base path}/api-gateway")
                   .build("service", "")
                   .toString(),
               equalTo("http://www.mulesoft.com/service/api-gateway"));

    assertThat(
               new HttpClientUriBuilder().fromUri("http://www.mulesoft.com").appendPath("/{base path}/{resource}")
                   .build("service", "api-gateway").toString(),
               equalTo("http://www.mulesoft.com/service/api-gateway"));

    assertThat(
               new HttpClientUriBuilder().fromUri("http://www.mulesoft.com").appendPath("/{base path}{api-gateway}")
                   .build("service", "api-gateway").toString(),
               equalTo("http://www.mulesoft.com/serviceapi-gateway"));
  }

  @Test
  public void nullPathShouldNotAffect() {
    assertThat(
               new HttpClientUriBuilder().fromUri("http://www.mulesoft.com/a/b?c=d&e=f").appendPath(null)
                   .build().toString(),
               equalTo("http://www.mulesoft.com/a/b?c=d&e=f"));
  }

  @Test
  public void nullQueryParamsShouldNotAffect() {
    assertThat(
               new HttpClientUriBuilder().fromUri("http://www.mulesoft.com/a/b").queryParam(null, "a")
                   .build().toString(),
               equalTo("http://www.mulesoft.com/a/b"));

    assertThat(
               new HttpClientUriBuilder().fromUri("http://www.mulesoft.com/a/b").queryParam("a", null)
                   .build().toString(),
               equalTo("http://www.mulesoft.com/a/b"));

    assertThat(
               new HttpClientUriBuilder().fromUri("http://www.mulesoft.com/a/b").queryParam(null, null)
                   .build().toString(),
               equalTo("http://www.mulesoft.com/a/b"));
  }

  @Test
  public void malFormedUrlShouldThrowAUriBuilderException() {
    expect(UriBuilderException.class, () -> new HttpClientUriBuilder()
        .fromUri("{}ll.[]")
        .build());

    expect(UriBuilderException.class, () -> new HttpClientUriBuilder()
        .fromUri("http://ll//.[]")
        .appendPath("/test")
        .build());

    expect(UriBuilderException.class, () -> new HttpClientUriBuilder()
        .fromUri("{}ll//.")
        .appendPath("/test[]")
        .build());

    expect(UriBuilderException.class, () -> new HttpClientUriBuilder()
        .fromUri("{}ll//.com")
        .appendPath("/test[]")
        .build());
  }



}
