/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.client.adapter;

import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;

import com.mulesoft.mule.runtime.gw.api.contract.NoSla;
import com.mulesoft.mule.runtime.gw.api.contract.Sla;
import com.mulesoft.mule.runtime.gw.api.contract.Contract;
import com.mulesoft.mule.runtime.gw.api.contract.tier.SingleTier;
import com.mulesoft.mule.runtime.gw.api.contract.tier.Tier;
import com.mulesoft.mule.runtime.gw.api.client.Client;
import com.mulesoft.mule.runtime.gw.client.dto.ContractDto;
import com.mulesoft.mule.runtime.gw.client.dto.LimitDto;
import com.mulesoft.mule.runtime.gw.api.construction.Builder;

import java.util.List;

public class ContractBuilder implements Builder<Contract> {

  private ContractDto dto;

  public ContractBuilder from(ContractDto dto) {
    this.dto = dto;
    return this;
  }

  @Override
  public Contract build() {
    Contract contract;

    Client client = Client.builder().withId(clientId()).withSecret(clientSecret()).withName(clientName()).build();

    if (!nullTier() && !nullLimits()) {
      Integer slaId = slaId();
      List<LimitDto> limits = dto.getTier().getLimits();

      List<Tier> tiers = limits
          .stream()
          .map(limit -> new SingleTier(limit.getMaximumRequests(), limit.getTimePeriodInMilliseconds()))
          .collect(toList());
      // TODO AGW-1169 Provide builders for all object in Gateway API
      contract = Contract.builder().withClient(client).withSla(new Sla(slaId, tiers)).build();
    } else {
      contract = Contract.builder().withClient(client).withSla(new NoSla()).build();
    }
    return contract;
  }

  private boolean nullLimits() {
    return dto.getTier().getLimits() == null;
  }

  private boolean nullTier() {
    return dto.getTier() == null;
  }

  private Integer slaId() {
    Integer slaId = dto.getTier().getId();
    requireNonNull(slaId, "Cannot create a Contract without an SLA ID");
    return slaId;
  }

  private String clientSecret() {
    String clientSecret = dto.getApp().getClientSecret();
    requireNonNull(clientSecret, "Cannot create a Contract without client secret");
    return clientSecret;
  }

  private String clientId() {
    String clientId = dto.getApp().getClientId();
    requireNonNull(clientId, "Cannot create a Contract without client id");
    return clientId;
  }

  private String clientName() {
    String clientName = dto.getApp().getName();
    requireNonNull(clientName, "Cannot create a Contract without client name");
    return clientName;
  }
}
