/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.client.httpclient;

import static com.google.common.collect.Lists.newArrayList;
import static java.util.Arrays.asList;
import static org.apache.commons.lang3.StringUtils.substringAfter;
import static org.apache.commons.lang3.StringUtils.substringBefore;
import static org.apache.http.auth.AuthScope.ANY_REALM;
import static org.apache.http.client.params.AuthPolicy.BASIC;
import static org.apache.http.client.params.AuthPolicy.DIGEST;
import static org.apache.http.client.params.AuthPolicy.NTLM;

import com.mulesoft.mule.runtime.gw.api.config.PlatformClientConfiguration;
import com.mulesoft.mule.runtime.gw.client.httpclient.connection.RestartableConnectionManager;
import com.mulesoft.mule.runtime.gw.client.httpclient.interceptors.HttpRequestResponseInterceptor;

import java.security.GeneralSecurityException;
import java.security.SecureRandom;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;

import org.apache.http.HttpHost;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.HttpResponseInterceptor;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.Credentials;
import org.apache.http.auth.NTCredentials;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.HttpClientConnectionManager;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;

public class GatewayHttpClientBuilder {

  private HttpClientBuilder clientBuilder;
  private RequestConfig.Builder requestConfigBuilder;
  private boolean sslValidationEnabled;

  public GatewayHttpClientBuilder() {
    clientBuilder = HttpClientBuilder.create();
    requestConfigBuilder = RequestConfig.custom();
  }

  public GatewayHttpClientBuilder withClientConfiguration(PlatformClientConfiguration clientConfiguration) {
    timeouts(clientConfiguration);
    setupProxy(clientConfiguration);

    return this;
  }

  public GatewayHttpClientBuilder withRequestResponseInterceptors(HttpRequestResponseInterceptor... interceptors) {
    asList(interceptors).forEach(httpRequestResponseInterceptor -> {
      clientBuilder.addInterceptorLast((HttpRequestInterceptor) httpRequestResponseInterceptor);
      clientBuilder.addInterceptorLast((HttpResponseInterceptor) httpRequestResponseInterceptor);
    });
    return this;
  }

  public GatewayHttpClientBuilder withSslValidationEnabled(boolean enabled) {
    this.sslValidationEnabled = enabled;
    return this;
  }

  public GatewayHttpClient build() {
    RestartableConnectionManager connectionManager = setupConnectionManager();
    RequestConfig requestConfig = requestConfigBuilder.build();
    clientBuilder.setDefaultRequestConfig(requestConfig);
    clientBuilder.setConnectionManager(connectionManager);

    return new GatewayHttpClient(clientBuilder.build(), connectionManager, requestConfig.getConnectTimeout() / 2);
  }

  private void timeouts(PlatformClientConfiguration timeoutConfiguration) {
    requestConfigBuilder.setSocketTimeout((int) timeoutConfiguration.getReadTimeout().inMillis());
    requestConfigBuilder.setConnectTimeout((int) timeoutConfiguration.getConnectTimeout().inMillis());
    requestConfigBuilder.setConnectionRequestTimeout((int) timeoutConfiguration.getConnectTimeout().inMillis());
  }

  protected SSLContext getSslContext() {
    TrustManager[] certs = new TrustManager[] {
        new X509TrustManager() {

          @Override
          public X509Certificate[] getAcceptedIssuers() {
            return null;
          }

          @Override
          public void checkServerTrusted(X509Certificate[] chain, String authType)
              throws CertificateException {}

          @Override
          public void checkClientTrusted(X509Certificate[] chain, String authType)
              throws CertificateException {}
        }
    };

    try {
      SSLContext ctx = SSLContext.getInstance("TLS");
      ctx.init(null, certs, new SecureRandom());
      return ctx;
    } catch (GeneralSecurityException ex) {
      throw new GatewayHttpClientBuildException("General Security Exception when initializing SSL context");
    }
  }

  protected RestartableConnectionManager setupConnectionManager() {
    try {
      if (!sslValidationEnabled) {
        return new RestartableConnectionManager(() -> new PoolingHttpClientConnectionManager(connectionSocketFactory()));
      } else {
        return new RestartableConnectionManager(PoolingHttpClientConnectionManager::new);
      }
    } catch (Exception e) {
      throw new GatewayHttpClientBuildException(e.getMessage());
    }
  }

  protected Registry<ConnectionSocketFactory> connectionSocketFactory() {
    return RegistryBuilder.<ConnectionSocketFactory>create()
        .register("https", new SSLConnectionSocketFactory(getSslContext(), NoopHostnameVerifier.INSTANCE))
        .register("http", PlainConnectionSocketFactory.getSocketFactory())
        .build();
  }

  private void setupProxy(PlatformClientConfiguration configuration) {
    if (configuration.getProxyHost() != null) {
      HttpHost proxy = new HttpHost(configuration.getProxyHost(), configuration.getProxyPort());

      requestConfigBuilder.setProxy(proxy);

      if (configuration.getProxyUserName() != null) {
        // This is to make HttpClient pick the Basic authentication scheme over Digest & NTLM
        requestConfigBuilder.setProxyPreferredAuthSchemes(newArrayList(BASIC, DIGEST, NTLM));

        AuthScope ntlmScope = new AuthScope(configuration.getProxyHost(), configuration.getProxyPort(), ANY_REALM, NTLM);
        AuthScope basicScope = new AuthScope(configuration.getProxyHost(), configuration.getProxyPort(), ANY_REALM, BASIC);
        Credentials ntCredentials = ntCredentials(configuration);
        Credentials basicCredentials = new UsernamePasswordCredentials(configuration.getProxyUserName(),
                                                                       configuration.getProxyPassword());

        CredentialsProvider credentialsProvider = new BasicCredentialsProvider();
        credentialsProvider.setCredentials(basicScope, basicCredentials);
        credentialsProvider.setCredentials(ntlmScope, ntCredentials);

        clientBuilder.setDefaultCredentialsProvider(credentialsProvider);
      }
    }
  }

  private NTCredentials ntCredentials(PlatformClientConfiguration configuration) {
    NTCredentials ntCredentials;

    if (configuration.getProxyUserName().contains("\\")) {
      String domain = substringBefore(configuration.getProxyUserName(), "\\");
      String username = substringAfter(configuration.getProxyUserName(), "\\");

      ntCredentials = new NTCredentials(username, configuration.getProxyPassword(), "", domain);
    } else {
      ntCredentials = new NTCredentials(configuration.getProxyUserName(), configuration.getProxyPassword(), "", "");
    }

    return ntCredentials;
  }
}
