/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.client.httpclient.connection;

import java.io.IOException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;

import org.apache.http.HttpClientConnection;
import org.apache.http.conn.ConnectionRequest;
import org.apache.http.conn.HttpClientConnectionManager;
import org.apache.http.conn.routing.HttpRoute;
import org.apache.http.protocol.HttpContext;

/**
 * A custom connection manager that will create a new connection manager when shutdown.
 */
public class RestartableConnectionManager implements HttpClientConnectionManager {

  private boolean isShutdown;
  private final AtomicReference<HttpClientConnectionManager> delegate;
  private final Supplier<HttpClientConnectionManager> supplier;

  public RestartableConnectionManager(Supplier<HttpClientConnectionManager> supplier) {
    this.supplier = supplier;
    delegate = new AtomicReference<>(supplier.get());
    isShutdown = false;
  }

  @Override
  public ConnectionRequest requestConnection(HttpRoute route, Object state) {
    return delegate.get().requestConnection(route, state);
  }

  @Override
  public void releaseConnection(HttpClientConnection conn, Object newState, long validDuration, TimeUnit timeUnit) {
    delegate.get().releaseConnection(conn, newState, validDuration, timeUnit);
  }

  @Override
  public void connect(HttpClientConnection conn, HttpRoute route, int connectTimeout, HttpContext context)
      throws IOException {
    delegate.get().connect(conn, route, connectTimeout, context);
  }

  @Override
  public void upgrade(HttpClientConnection conn, HttpRoute route, HttpContext context) throws IOException {
    delegate.get().upgrade(conn, route, context);
  }

  @Override
  public void routeComplete(HttpClientConnection conn, HttpRoute route, HttpContext context) throws IOException {
    delegate.get().routeComplete(conn, route, context);
  }

  @Override
  public void closeIdleConnections(long idletime, TimeUnit timeUnit) {
    delegate.get().closeIdleConnections(idletime, timeUnit);
  }

  @Override
  public void closeExpiredConnections() {
    delegate.get().closeExpiredConnections();
  }

  @Override
  public void shutdown() {
    synchronized (delegate) {
      delegate.get().shutdown();
      isShutdown = true;
    }
  }

  public void restart() {
    synchronized (delegate) {
      if (isShutdown) {
        delegate.set(supplier.get());
        isShutdown = false;
      }
    }
  }

}
