/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.client.httpclient.interceptors;

import static com.fasterxml.jackson.databind.DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES;
import static com.mulesoft.mule.runtime.gw.api.logging.ExceptionDescriptor.errorMessage;
import static com.mulesoft.mule.runtime.gw.client.ApiPlatformClient.GATEWAY_API_BASE_RESOURCE;
import static com.mulesoft.mule.runtime.gw.client.ApiPlatformClient.PROXIES_API_BASE_RESOURCE;
import static org.apache.http.entity.ContentType.APPLICATION_JSON;
import static org.mule.runtime.core.api.config.MuleManifest.getProductVersion;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.PROXY_AUTHENTICATION_REQUIRED;
import static org.mule.runtime.http.api.HttpHeaders.Names.AUTHORIZATION;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;

import org.apache.http.HttpEntity;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.auth.Credentials;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.entity.StringEntity;
import org.apache.http.protocol.HttpContext;

import org.mule.runtime.core.api.util.IOUtils;

import org.slf4j.Logger;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import com.mulesoft.mule.runtime.gw.client.exception.EntityParsingException;
import com.mulesoft.mule.runtime.gw.client.exception.EntityUnparsingException;
import com.mulesoft.mule.runtime.gw.client.exception.HttpConnectionException;
import com.mulesoft.mule.runtime.gw.client.exception.HttpResponseException;
import com.mulesoft.mule.runtime.gw.client.exception.UnauthorizedException;
import com.mulesoft.mule.runtime.gw.client.model.PortalAuthentication;
import com.mulesoft.mule.runtime.gw.logging.GatewayMuleAppLoggerFactory;

public abstract class Oauth2Interceptor implements HttpRequestResponseInterceptor, ResettableInterceptor {

  private static final String ACCESS_TOKEN_KEY = "com.mulesoft.module.client.httpclient.interceptors.accessToken";

  private final URI loginUrl;
  private final ObjectMapper objectMapper;
  private final HttpClient client;
  private final AccessTokenManager tokenManager;

  protected Oauth2Interceptor(HttpClient client, URI loginUrl) {
    this.client = client;
    this.loginUrl = loginUrl;
    this.objectMapper = new ObjectMapper();
    this.objectMapper.configure(FAIL_ON_UNKNOWN_PROPERTIES, false);
    this.tokenManager = new AccessTokenManager();
  }

  @Override
  public void process(HttpRequest request, HttpContext context) {
    AccessTokens tokens = tokenManager.getUpdatedTokens(this::authenticate, this::exchangeToken);

    if (request.containsHeader(AUTHORIZATION)) {
      request.removeHeaders(AUTHORIZATION);
    }

    HttpClientContext clientContext = (HttpClientContext) context;

    String authorizationHeaderToken = request.getRequestLine().getUri().contains(GATEWAY_API_BASE_RESOURCE)
        || request.getRequestLine().getUri().contains(PROXIES_API_BASE_RESOURCE) ? tokens.getPlatformToken()
            : tokens.getCoreServicesToken();

    if (authorizationHeaderToken != null) {
      Credentials credentials = new UsernamePasswordCredentials(authorizationHeaderToken, null);
      clientContext.getTargetAuthState().update(new GatewayAuthScheme(), credentials);
    }

    context.setAttribute(ACCESS_TOKEN_KEY, tokens.getCoreServicesToken());
  }

  @Override
  public void process(HttpResponse response, HttpContext context) {
    String accessTokenInUse = (String) context.getAttribute(ACCESS_TOKEN_KEY);

    int statusCode = response.getStatusLine().getStatusCode();
    if (statusCode == 403 || statusCode == 401) {
      tokenManager.resetIfEquals(accessTokenInUse);
    }
  }

  @Override
  public void reset() {
    tokenManager.reset();
  }

  public abstract String authenticate();

  String exchangeToken(String accessToken) {
    try {
      PortalAuthentication portalAuthentication = new PortalAuthentication();
      portalAuthentication.setToken(accessToken);
      portalAuthentication.setApiGatewayVersion(getProductVersion());

      HttpResponse response = executePost(loginUrl,
                                          new StringEntity(objectMapper.writeValueAsString(portalAuthentication),
                                                           APPLICATION_JSON));

      return processResponse(response, PortalAuthentication.class).getToken();
    } catch (JsonProcessingException e) {
      throw new EntityParsingException("Error building HTTP Request entity", e);
    }
  }

  protected HttpResponse executePost(URI uri, HttpEntity entity) {
    try {
      HttpPost post = new HttpPost(uri);
      post.setEntity(entity);
      return client.execute(post);
    } catch (IOException e) {
      throw new HttpConnectionException("An error occurred executing OAuth HTTP Request. " + errorMessage(e), e);
    }
  }

  protected <T> T processResponse(HttpResponse response, Class<T> clazz) {
    try (InputStream responseInputStream = response.getEntity().getContent()) {
      if (response.getStatusLine().getStatusCode() != 200) {
        String message = response.getStatusLine().getStatusCode() != PROXY_AUTHENTICATION_REQUIRED.getStatusCode()
            ? "Authorization request to Anypoint Platform was not successful, client_id and/or client_secret may be wrong."
            : "Proxy authorization request failed, configured proxy username and/or password may be wrong.";

        throw new UnauthorizedException(message, IOUtils.toString(responseInputStream), response.getStatusLine().getStatusCode());
      }

      return readResponseValue(response, clazz);
    } catch (IOException e) {
      throw new HttpResponseException("An error occurred processing OAuth HTTP Response. " + errorMessage(e), e);
    }
  }

  protected <T> T readResponseValue(HttpResponse response, Class<T> clazz) {
    try {
      return objectMapper.readValue(response.getEntity().getContent(), clazz);
    } catch (IOException e) {
      throw new EntityUnparsingException("Unable to parse HTTP Response content. " + errorMessage(e), e,
                                         response.getStatusLine().getStatusCode());
    }
  }

}
