/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.client;

import static com.fasterxml.jackson.databind.DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES;
import static com.fasterxml.jackson.databind.DeserializationFeature.READ_ENUMS_USING_TO_STRING;
import static com.fasterxml.jackson.databind.SerializationFeature.WRITE_ENUMS_USING_TO_STRING;
import static com.google.common.collect.Lists.newArrayList;
import static com.mulesoft.mule.runtime.gw.api.logging.ExceptionDescriptor.errorMessage;
import static com.mulesoft.mule.runtime.gw.client.ApiPlatformClient.RestClientStatus.INITIALISED;
import static com.mulesoft.mule.runtime.gw.client.ApiPlatformClient.RestClientStatus.NOT_INITIALISED;
import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toList;
import static org.apache.http.HttpHeaders.CONTENT_ENCODING;
import static org.apache.http.HttpHeaders.IF_NONE_MATCH;
import static org.apache.http.entity.ContentType.APPLICATION_JSON;
import static org.apache.http.entity.ContentType.create;
import static org.apache.http.util.EntityUtils.consumeQuietly;
import static org.mule.runtime.core.api.config.MuleManifest.getProductVersion;
import static org.mule.runtime.http.api.HttpHeaders.Names.ETAG;
import static org.mule.runtime.http.api.HttpHeaders.Values.GZIP;

import com.mulesoft.mule.runtime.gw.api.analytics.AnalyticsHttpEvent;
import com.mulesoft.mule.runtime.gw.api.config.GatewayConfiguration;
import com.mulesoft.mule.runtime.gw.api.config.PlatformClientConfiguration;
import com.mulesoft.mule.runtime.gw.api.config.PlatformUnauthenticatedClientConfiguration;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.api.policy.PolicyTemplateKey;
import com.mulesoft.mule.runtime.gw.client.adapter.ApiClientsResponseBuilder;
import com.mulesoft.mule.runtime.gw.client.adapter.ApiResponseBuilder;
import com.mulesoft.mule.runtime.gw.client.dto.ApiClientDto;
import com.mulesoft.mule.runtime.gw.client.dto.ApiDto;
import com.mulesoft.mule.runtime.gw.client.dto.CoreServicesClientDto;
import com.mulesoft.mule.runtime.gw.client.dto.PolicyTemplateDto;
import com.mulesoft.mule.runtime.gw.client.exception.EntityParsingException;
import com.mulesoft.mule.runtime.gw.client.exception.EntityUnparsingException;
import com.mulesoft.mule.runtime.gw.client.exception.HttpConnectionException;
import com.mulesoft.mule.runtime.gw.client.exception.HttpResponseException;
import com.mulesoft.mule.runtime.gw.client.httpclient.GatewayHttpClient;
import com.mulesoft.mule.runtime.gw.client.httpclient.GatewayHttpClientBuilder;
import com.mulesoft.mule.runtime.gw.client.httpclient.interceptors.BenchmarkInterceptor;
import com.mulesoft.mule.runtime.gw.client.httpclient.interceptors.HttpRequestResponseInterceptor;
import com.mulesoft.mule.runtime.gw.client.httpclient.interceptors.HttpResponseStatusInterceptor;
import com.mulesoft.mule.runtime.gw.client.httpclient.interceptors.Oauth2ClientCredentialsInterceptor;
import com.mulesoft.mule.runtime.gw.client.httpclient.interceptors.TraceInterceptor;
import com.mulesoft.mule.runtime.gw.client.httpclient.interceptors.UserAgentEnricherInterceptor;
import com.mulesoft.mule.runtime.gw.client.model.ApiClientsResponse;
import com.mulesoft.mule.runtime.gw.client.model.ApiResponse;
import com.mulesoft.mule.runtime.gw.client.model.ClientRows;
import com.mulesoft.mule.runtime.gw.client.model.HttpEvent;
import com.mulesoft.mule.runtime.gw.client.model.HttpEventViews;
import com.mulesoft.mule.runtime.gw.client.model.Me;
import com.mulesoft.mule.runtime.gw.client.response.PlatformResponse;
import com.mulesoft.mule.runtime.gw.client.uri.HttpClientUriBuilder;
import com.mulesoft.mule.runtime.gw.logging.GatewayMuleAppLoggerFactory;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Scanner;
import java.util.zip.GZIPOutputStream;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.message.BasicNameValuePair;
import org.slf4j.Logger;

public class ApiPlatformClient {

  // API Platform Resources
  public static final String GATEWAY_API_BASE_RESOURCE = "/apigateway/apimanager/v1";
  public static final String PROXIES_API_BASE_RESOURCE = "/proxies/xapi/v1";
  private static final Logger LOGGER = GatewayMuleAppLoggerFactory.getLogger(ApiPlatformClient.class);
  private static final String LOGIN_RESOURCE = "/apiplatform/login";
  private static final String APIS_RESOURCE = GATEWAY_API_BASE_RESOURCE + "/apis/{apiId}";
  private static final String POLICY_TEMPLATE_RESOURCE = GATEWAY_API_BASE_RESOURCE + "/templates";
  private static final String ACTIVE_ENDPOINT_RESOURCE =
      PROXIES_API_BASE_RESOURCE + "/organizations/{organizationId}/environments/{environmentId}/apis/{environmentApiId}/active";

  // Core-Services Resources
  private static final String AUTH_RESOURCE = "/accounts/oauth2/token";
  private static final String ORGANIZATION_CLIENT_RESOURCE = "/accounts/api/organizations/{orgId}/clients/{clientId}";
  private static final String ME_RESOURCE = "/accounts/api/me";

  // Caching Contracts Service Resources
  private static final String API_CONTRACTS_RESOURCE =
      "/apigateway/ccs/v2/organizations/{orgId}/environments/{environmentId}/apis/{environmentApiId}/contracts";

  // Analytics Resources
  private static final String ANALYTICS_EVENTS_RESOURCE = "/v2/analytics/events";
  private static final String METRICS_EVENTS_RESOURCE = "/apiruntime/v1/events";

  private final ObjectMapper objectMapper;

  private String platformBaseUri;
  private String analyticsBaseUri;
  private String metricsBaseUri;
  private String policyTemplateSource;

  private GatewayHttpClient client;
  private GatewayHttpClient unauthenticatedClient;
  private GatewayHttpClient metricsClient;

  private PlatformClientConfiguration configuration;

  // cache
  private String orgId;
  private RestClientStatus status = NOT_INITIALISED;

  public ApiPlatformClient() {
    // TODO AGW-1561: wrap ObjectMapper into a proper object
    this.objectMapper =
        new ObjectMapper()
            .configure(FAIL_ON_UNKNOWN_PROPERTIES, false)
            .enable(READ_ENUMS_USING_TO_STRING)
            .enable(WRITE_ENUMS_USING_TO_STRING);

    this.objectMapper.setConfig(objectMapper.getSerializationConfig().withView(HttpEventViews.Public.class));
  }

  public void configure(GatewayConfiguration gatewayConfiguration) throws URISyntaxException {
    this.configuration = gatewayConfiguration.platformClient();

    this.platformBaseUri = configuration.getPlatformUri();
    this.analyticsBaseUri = configuration.getAnalyticsUri();
    this.metricsBaseUri = configuration.getMetricsBaseUri();
    this.policyTemplateSource = configuration.getPolicyTemplateSource();

    TraceInterceptor traceInterceptor = new TraceInterceptor(LOGGER);
    BenchmarkInterceptor benchmarkInterceptor = new BenchmarkInterceptor(LOGGER);

    URI oauthTokenUri = new URI(this.platformBaseUri + AUTH_RESOURCE);
    URI loginUri = new URI(this.platformBaseUri + LOGIN_RESOURCE);

    HttpRequestResponseInterceptor oauth2Interceptor =
        new Oauth2ClientCredentialsInterceptor(buildClient(traceInterceptor, benchmarkInterceptor), oauthTokenUri,
                                               loginUri, configuration.getClientId(),
                                               configuration.getClientSecret());

    client = buildClient(oauth2Interceptor, new UserAgentEnricherInterceptor(), traceInterceptor, benchmarkInterceptor,
                         new HttpResponseStatusInterceptor());

    unauthenticatedClient =
        buildClient(new PlatformUnauthenticatedClientConfiguration(), configuration.enableSSLValidation(),
                    new UserAgentEnricherInterceptor(), traceInterceptor, benchmarkInterceptor,
                    new HttpResponseStatusInterceptor());

    metricsClient =
        buildClient(new PlatformUnauthenticatedClientConfiguration(), configuration.enableSSLValidation(),
                    new UserAgentEnricherInterceptor(), traceInterceptor, benchmarkInterceptor,
                    new HttpResponseStatusInterceptor());
  }

  private GatewayHttpClient buildClient(HttpRequestResponseInterceptor... interceptors) {
    return buildClient(configuration, interceptors);
  }

  private GatewayHttpClient buildClient(PlatformClientConfiguration clientConfiguration,
                                        HttpRequestResponseInterceptor... interceptors) {
    return buildClient(clientConfiguration, clientConfiguration.enableSSLValidation(), interceptors);
  }

  private GatewayHttpClient buildClient(PlatformClientConfiguration clientConfiguration, boolean sslValidationEnabled,
                                        HttpRequestResponseInterceptor... interceptors) {
    return new GatewayHttpClientBuilder()
        .withClientConfiguration(clientConfiguration)
        .withRequestResponseInterceptors(interceptors)
        .withSslValidationEnabled(sslValidationEnabled)
        .build();
  }

  public void cleanConnections() {
    client.cleanConnections();
  }

  public PlatformResponse<ApiClientsResponse> getApiClients(String organizationId, String environmentId, Long apiId,
                                                            String contractsEntityTag) {
    URI uri = new HttpClientUriBuilder().fromUri(platformBaseUri)
        .appendPath(API_CONTRACTS_RESOURCE)
        .build(organizationId, environmentId, apiId);

    HttpGet request = new HttpGet(uri);

    if (contractsEntityTag != null) {
      request.setHeader(IF_NONE_MATCH, contractsEntityTag);
    }

    HttpResponse response = doExecute(request);

    assertStatusCodes("Unable to retrieve contracts for " + apiId, response, 200, 304);

    int statusCode = statusCode(response);

    ApiClientsResponseBuilder apiClientsResponseBuilder = new ApiClientsResponseBuilder();

    if (statusCode == 200) {
      apiClientsResponseBuilder.withClients(parseClients(response));

      if (response.getFirstHeader(ETAG) != null) {
        apiClientsResponseBuilder.withContractsEntityTag(response.getFirstHeader(ETAG).getValue());
      }
    } else {
      consumeQuietly(response.getEntity());
      apiClientsResponseBuilder.noUpdates();
    }

    return new PlatformResponse<>(apiClientsResponseBuilder.build(), statusCode(response));
  }

  public PlatformResponse activateEndpoint(String organizationId, String environmentId, Long apiId) {
    URI uri = new HttpClientUriBuilder().fromUri(platformBaseUri)
        .appendPath(ACTIVE_ENDPOINT_RESOURCE)
        .build(organizationId, environmentId, apiId);

    HttpEntityEnclosingRequestBase post = new HttpPost(uri);
    post.setEntity(emptyJsonEntity());

    HttpResponse response = doExecute(post);
    consumeQuietly(response.getEntity());

    assertStatusCodes("Unable to activate endpoint for " + apiId, response, 201);

    return new PlatformResponse<>(null, statusCode(response));
  }

  public PlatformResponse<ApiResponse> getApi(ApiKey apiKey, String apiEntityTag) {
    URI uri = new HttpClientUriBuilder().fromUri(platformBaseUri)
        .appendPath(APIS_RESOURCE)
        .queryParam("version", getProductVersion())
        .build(apiKey.id());

    HttpGet request = new HttpGet(uri);

    if (apiEntityTag != null) {
      request.setHeader(IF_NONE_MATCH, apiEntityTag);
    }

    HttpResponse response = doExecute(request);

    assertStatusCodes("Unable to get API information from API Platform", response, 200, 304);

    int statusCode = statusCode(response);

    ApiResponseBuilder apiResponseBuilder = new ApiResponseBuilder();

    if (statusCode == 200) {
      apiResponseBuilder.withApi(parseEntity(response, ApiDto.class));

      if (response.getFirstHeader(ETAG) != null) {
        apiResponseBuilder.withApiEntityTag(response.getFirstHeader(ETAG).getValue());
      }
    } else {
      consumeQuietly(response.getEntity());
      apiResponseBuilder.noUpdates();
    }

    return new PlatformResponse<>(apiResponseBuilder.build(), statusCode);
  }

  public int postHttpEvents(List<AnalyticsHttpEvent> events) throws IOException {
    LOGGER.trace("Sending events to analytics ingest: {}", events);

    URI uri = new HttpClientUriBuilder().fromUri(analyticsBaseUri)
        .appendPath(ANALYTICS_EVENTS_RESOURCE)
        .build();

    ObjectWriter writer = objectMapper.writer();

    if (configuration.isOnPrem()) {
      writer = objectMapper.writerWithView(HttpEventViews.ExternalAnalytics.class);
    }

    ByteArrayEntity entity;

    try {
      ByteArrayOutputStream baos = new ByteArrayOutputStream();
      GZIPOutputStream gzos = new GZIPOutputStream(baos);

      List<HttpEvent> eventsDto = events.stream().map(HttpEvent::from).collect(toList());
      writer.writeValue(gzos, eventsDto);

      ContentType contentType = create("application/vnd.mulesoft.analytics.http-event+json", new BasicNameValuePair("v", "2"));
      entity = new ByteArrayEntity(baos.toByteArray(), contentType);
    } catch (IOException e) {
      throw new EntityParsingException("Unable to parse HTTP Request content. " + errorMessage(e), e);
    }

    HttpPost postMethod = new HttpPost(uri);
    postMethod.setHeader(CONTENT_ENCODING, GZIP);
    postMethod.setEntity(entity);

    HttpResponse response = client.execute(postMethod);
    consumeQuietly(response.getEntity());

    return statusCode(response);
  }

  public int postMetrics(String eventsBody) throws IOException {
    URI uri = new HttpClientUriBuilder()
        .fromUri(metricsBaseUri)
        .appendPath(METRICS_EVENTS_RESOURCE)
        .build();

    ContentType contentType = create("application/json");
    ByteArrayEntity entity = new ByteArrayEntity(eventsBody.getBytes(StandardCharsets.UTF_8), contentType);

    HttpPost postMethod = new HttpPost(uri);
    postMethod.setEntity(entity);
    HttpResponse response = metricsClient.execute(postMethod);
    consumeQuietly(response.getEntity());
    return statusCode(response);
  }

  public PlatformResponse<Me> connect() {
    URI uri = new HttpClientUriBuilder().fromUri(platformBaseUri)
        .appendPath(ME_RESOURCE)
        .build();

    PlatformResponse<Me> response = execute(new HttpGet(uri), Me.class);
    Me me = response.entity();
    orgId = me.getUser().getOrganization().getId();
    status = INITIALISED;
    return new PlatformResponse<>(me, response.statusCode());
  }

  public PolicyTemplateDto getPolicyTemplateMetadata(PolicyTemplateKey templateKey) {
    URI uri = new HttpClientUriBuilder().fromUri(platformBaseUri)
        .appendPath(POLICY_TEMPLATE_RESOURCE)
        .queryParam("muleVersion", getProductVersion())
        .queryParam("groupId", templateKey.getGroupId())
        .queryParam("assetId", templateKey.getAssetId())
        .queryParam("version", templateKey.getVersion())
        .queryParam("source", policyTemplateSource)
        .build();

    return execute(new HttpGet(uri), PolicyTemplateDto.class).entity();
  }

  public InputStream downloadTemplateAsset(String link) throws URISyntaxException, IOException {
    URI uri = new HttpClientUriBuilder().fromUri(link)
        .build();

    return unauthenticatedClient.execute(new HttpGet(uri)).getEntity().getContent();
  }

  public String getOrgId() {
    return orgId;
  }

  public boolean isConnected() {
    return INITIALISED.equals(status);
  }

  public void shutdown() {
    if (client != null) {
      client.shutdown();
    }
  }

  private void assertStatusCodes(String message, HttpResponse response, Integer... validStatusCodes) {
    if (!asList(validStatusCodes).contains(statusCode(response))) {
      consumeQuietly(response.getEntity());
      throw new HttpResponseException(message, statusCode(response));
    }
  }

  private List<ApiClientDto> parseClients(HttpResponse response) {
    try (Scanner scanner = new Scanner(content(response))) {
      readCountRow(scanner);
      return getApiClients(scanner);
    } catch (IOException e) {
      throw new EntityUnparsingException("Unable to parse HTTP Response content. " + errorMessage(e), e, statusCode(response));
    }
  }

  private ClientRows readCountRow(Scanner scanner) throws IOException {
    String header = scanner.nextLine();
    return objectMapper.readValue(header, ClientRows.class);
  }

  private List<ApiClientDto> getApiClients(Scanner scanner) throws IOException {
    List<ApiClientDto> clients = newArrayList();

    while (scanner.hasNextLine()) {
      clients.add(objectMapper.readValue(scanner.nextLine(), ApiClientDto.class));
    }

    return clients;
  }

  public PlatformResponse<CoreServicesClientDto> getClient(String id, String secret) {
    URI uri = new HttpClientUriBuilder().fromUri(platformBaseUri)
        .appendPath(ORGANIZATION_CLIENT_RESOURCE)
        .build(getOrgId(), id);

    HttpGet request = new HttpGet(uri);
    request.addHeader("clientSecret", secret);

    return execute(request, CoreServicesClientDto.class);
  }

  private int statusCode(HttpResponse response) {
    return response.getStatusLine().getStatusCode();
  }

  private <T> T parseEntity(HttpResponse response, Class<T> valueType) {
    try {
      return objectMapper.readValue(content(response), valueType);
    } catch (IOException e) {
      throw new EntityUnparsingException("Unable to parse HTTP Response content. " + errorMessage(e), e, statusCode(response));
    }
  }

  private InputStream content(HttpResponse response) {
    try {
      return response.getEntity().getContent();
    } catch (IOException e) {
      throw new HttpConnectionException("Unable to read HTTP Response content. " + errorMessage(e), e);
    }
  }

  private HttpEntity emptyJsonEntity() {
    return new StringEntity("", APPLICATION_JSON);
  }

  private <T> PlatformResponse<T> execute(HttpUriRequest request, Class<T> valueType) {
    HttpResponse response = doExecute(request);
    return new PlatformResponse<>(parseEntity(response, valueType), statusCode(response));
  }

  private HttpResponse doExecute(HttpUriRequest request) {
    try {
      return client.execute(request);
    } catch (IOException e) {
      throw new HttpConnectionException("Unable to execute HTTP Request. " + errorMessage(e), e);
    }
  }

  public enum RestClientStatus {

    NOT_INITIALISED, INITIALISED

  }
}
