/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.client.uri;

import static java.util.Optional.ofNullable;

import com.mulesoft.mule.runtime.gw.client.exception.UriBuilderException;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;

import org.apache.http.NameValuePair;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.message.BasicNameValuePair;

public class HttpClientUriBuilder implements UriBuilder {

  private String baseUri = "";
  private String path = "";
  private final List<NameValuePair> queryParams = new LinkedList<>();

  public HttpClientUriBuilder() {}

  public HttpClientUriBuilder fromUri(String baseUri) {
    if (Objects.nonNull(baseUri)) {
      this.baseUri = baseUri;
    }

    return this;
  }

  public HttpClientUriBuilder appendPath(String path) {
    if (Objects.nonNull(path)) {
      this.path = appendPath(this.path, path);
    }

    return this;
  }

  public HttpClientUriBuilder queryParam(String name, String value) {
    if (Objects.nonNull(name) && Objects.nonNull(value)) {
      queryParams.add(new BasicNameValuePair(name, value));
    }

    return this;
  }

  public URI build(Object... params) {
    String parametrizedPath = replaceParams(path, params);

    try {
      URIBuilder builder = new URIBuilder(this.baseUri);

      if (!parametrizedPath.isEmpty() || !queryParams.isEmpty()) {
        builder.setPath(appendPath(ofNullable(builder.getPath()).orElse("/"), parametrizedPath));
      }

      if (!queryParams.isEmpty()) {
        builder.setParameters(queryParams);
      }

      return builder.build();

    } catch (URISyntaxException e) {
      throw new UriBuilderException(e);
    }
  }

  private String appendPath(String base, String append) {
    return base.endsWith("/") && append.startsWith("/") ? base + append.substring(1) : base + append;
  }

  private String replaceParams(String target, Object... params) {
    return String.format(target.replaceAll("\\{.*?\\}", "%s"), params);
  }
}
