/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.api.service;

import org.mule.runtime.api.service.Service;

import com.mulesoft.mule.runtime.gw.api.ApiContracts;
import com.mulesoft.mule.runtime.gw.api.ApiContractsSupplier;
import com.mulesoft.mule.runtime.gw.api.contract.Contract;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.api.service.exception.UnknownAPIException;

import java.util.List;

/**
 * Defines the ContractService, who provides the up-to-date {@link Contract}s for tracked {@link ApiContracts}s.
 */
public interface ContractService extends Service {

  /**
   * The {@link ContractService} delegates the maintaining of up-to-date {@link ApiContracts}s to an {@link ApiContractsSupplier}. Every information
   * regarding them, will be forwarded to the {@link ApiContractsSupplier}.
   * 
   * @param supplier the {@link ApiContractsSupplier}.
   * @return this.
   */
  ContractService contractSupplier(ApiContractsSupplier supplier);

  /**
   * The {@link ContractService} delegates the pre fetching of the {@link ApiContracts}s to an {@link ApiContractsPrefetch}. Every information
   * regarding them, will be forwarded to the {@link ApiContractsSupplier}.
   *
   * @param prefetch the {@link ApiContractsPrefetch}.
   * @return this.
   */
  default ContractService contractPrefetch(ApiContractsPrefetch prefetch) {
    return this;
  }

  /**
   * @return the current {@link ApiContracts} matching the {@link ApiKey}, raises an exception when not found.
   * @throws UnknownAPIException
   */
  ApiContracts contracts(ApiKey apiKey) throws UnknownAPIException;

  /**
   * Notifies the {@link ContractService} that someone needs to track an {@link ApiContracts} with matching key, therefore its contracts
   * must be fetched. A description of the tracker is only added for detailed logging.
   * 
   * @param key {@link ApiKey}.
   * @param trackerDescription {@link String} describing the tracker.
   * @return this.
   */
  ContractService track(ApiKey key, String trackerDescription);

  /**
   * Notifies the {@link ContractService} that someone is not tracking an {@link ApiContracts} anymore, if this {@link Service} decides to,
   * can stop fetching {@link Contract}s for it. Implementation should only stop fetching them when everyone has stopped tracking
   * it.
   *
   * @param key {@link ApiKey}.
   * @param trackerDescription {@link String} describing the tracker.
   * @return this.
   */
  ContractService untrack(ApiKey key, String trackerDescription);

  /**
   * @return the current tracked Api's {@link ApiKey}, every Api in this list will be requiring {@link Contract}s.
   */
  List<ApiKey> trackedApis();

}
