/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.gateway.service;

import static java.util.Optional.ofNullable;

import com.mulesoft.mule.runtime.gw.api.ApiContracts;
import com.mulesoft.mule.runtime.gw.api.ApiContractsSupplier;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.api.service.ApiContractsPrefetch;
import com.mulesoft.mule.runtime.gw.api.service.ContractService;
import com.mulesoft.mule.runtime.gw.api.service.exception.UnknownAPIException;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContractServiceImplementation implements ContractService {

  private static final Logger LOGGER = LoggerFactory.getLogger(ContractServiceImplementation.class);

  private ApiContractsSupplier apiContractsSupplier;
  private ApiContractsPrefetch apiContractsPrefetch;

  private final List<ApiKey> apisTrackedForContracts = new ArrayList<>();

  @Override
  public ContractService contractSupplier(ApiContractsSupplier supplier) {
    this.apiContractsSupplier = supplier;
    return this;
  }

  @Override
  public ContractService contractPrefetch(ApiContractsPrefetch prefetch) {
    this.apiContractsPrefetch = prefetch;
    return this;
  }

  @Override
  public ApiContracts contracts(ApiKey apiKey) throws UnknownAPIException {
    checkSupplierNonNull(apiKey);
    return apiContractsSupplier
        .getContracts(apiKey)
        .orElseThrow(() -> new UnknownAPIException("ContractService is not tracking " + apiKey + "."));
  }

  @Override
  public ContractService track(ApiKey key, String trackerDescription) {
    apisTrackedForContracts.add(key);
    LOGGER.debug("API {} expects contracts as it is used by {}.", key.id(), trackerDescription);
    ofNullable(apiContractsPrefetch).ifPresent(supplier -> apiContractsPrefetch.contractsRequired(key));
    return this;
  }

  @Override
  public ContractService untrack(ApiKey key, String trackerDescription) {
    if (!apisTrackedForContracts.contains(key)) {
      LOGGER.error("Trying to untrack Api: {} that hasn't been tracked.", key);
    } else {
      apisTrackedForContracts.remove(key);
      if (!apisTrackedForContracts.contains(key)) {
        ofNullable(apiContractsPrefetch).ifPresent(supplier -> apiContractsPrefetch.noContractsRequired(key));
        LOGGER.debug("Tracker {} does not need contracts for API {} anymore.", trackerDescription, key.id());
      } else {
        LOGGER.debug("After untrack, tracker {} has still at least one instance that need contracts for API {}.",
                     trackerDescription, key.id());
      }
    }
    return this;
  }

  @Override
  public List<ApiKey> trackedApis() {
    return apisTrackedForContracts;
  }

  @Override
  public String getName() {
    return "ContractService";
  }

  private void checkSupplierNonNull(ApiKey apiKey) throws UnknownAPIException {
    if (apiContractsSupplier == null) {
      throw new UnknownAPIException("Cannot lookup " + apiKey + " as there is no ApiContractsSupplier");
    }
  }

}
