/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.gateway.service;

import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY_2;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideVariable;
import static java.util.Arrays.asList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasItem;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;
import static org.junit.rules.ExpectedException.none;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_HOME_DIRECTORY_PROPERTY;

import org.mule.tck.junit4.rule.SystemPropertyTemporaryFolder;

import com.mulesoft.anypoint.tests.logger.DebugLine;
import com.mulesoft.anypoint.tests.logger.ErrorLine;
import com.mulesoft.anypoint.tests.logger.MockLogger;
import com.mulesoft.mule.runtime.gw.api.ApiContracts;
import com.mulesoft.mule.runtime.gw.api.client.Client;
import com.mulesoft.mule.runtime.gw.api.contract.Contract;
import com.mulesoft.mule.runtime.gw.api.contract.Sla;
import com.mulesoft.mule.runtime.gw.api.contract.tier.SingleTier;
import com.mulesoft.mule.runtime.gw.api.exception.ForbiddenClientException;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.api.service.ContractService;
import com.mulesoft.mule.runtime.gw.api.service.exception.UnknownAPIException;

import java.util.ArrayList;
import java.util.List;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;

public class ContractServiceTestCase {

  private static final String TRACKER = "My trekking boots";
  @Rule
  public ExpectedException thrown = none();

  @Rule
  public TemporaryFolder muleHome = new SystemPropertyTemporaryFolder(MULE_HOME_DIRECTORY_PROPERTY);

  private ContractService service;

  private MockApiContractsSupplier contractSupplier;
  private MockLogger mockLogger;

  @Before
  public void setUp() throws NoSuchFieldException, IllegalAccessException {
    this.mockLogger = new MockLogger();
    this.contractSupplier = new MockApiContractsSupplier();
    this.service = new ContractServiceImplementation()
        .contractSupplier(contractSupplier)
        .contractPrefetch(contractSupplier);
    overrideVariable("LOGGER").in(service).with(mockLogger);
  }

  @Test
  public void apiNotTrackedByServiceRaisesException() throws UnknownAPIException {
    expectedUnknownAPI();

    service.contracts(key());
  }

  @Test
  public void createAPIWithNoContracts() throws UnknownAPIException {
    contractSupplier.set(key(), noContracts());
    ApiContracts obtainedAPIContracts = service.contracts(key());

    assertEquals("Api should not have any clients", contractSupplier.getContracts(key()).get(), obtainedAPIContracts);
  }

  @Test
  public void createAPIWithMultipleContracts() throws ForbiddenClientException, UnknownAPIException {
    contractSupplier.set(key(), allContracts());
    ApiContracts obtainedAPIContracts = service.contracts(key());

    obtainedAPIContracts.validate(client().id(), client().secret());
    obtainedAPIContracts.validate(aThirdClient().id(), aThirdClient().secret());
    obtainedAPIContracts.validate(anotherClient().id(), anotherClient().secret());

    assertClientNotValid(obtainedAPIContracts, unknownClient());
  }

  @Test
  public void requestInexistentApi() throws UnknownAPIException {
    thrown.expect(UnknownAPIException.class);
    service.contracts(key());
  }

  @Test
  public void notifyServiceAPIsThatContainContracts() {
    service.track(key(), TRACKER);

    assertThat(service.trackedApis(), hasItem(key()));
    assertThat(service.trackedApis(), not(hasItem(anotherKey())));
    assertThat(mockLogger.lines(), hasSize(1));
    assertThat(mockLogger.lines().get(0), is(trackerAddedDebugLine(key())));
  }

  @Test
  public void contractTrackingFiresPrefetch() {
    service.track(key(), TRACKER);

    assertThat(contractSupplier.prefetchedKeys(), hasItem(key()));
  }

  @Test
  public void trackUntrack() {
    service.track(key(), TRACKER);
    service.untrack(key(), TRACKER);

    assertThat(contractSupplier.prefetchedKeys(), not(hasItem(key())));
  }

  @Test
  public void multipleTrackSingleUntrack() {
    service.track(key(), TRACKER);
    service.track(key(), TRACKER);
    service.untrack(key(), TRACKER);

    assertThat(contractSupplier.prefetchedKeys(), hasItem(key()));
  }

  @Test
  public void multipleTrackMultipleUntrack() {
    service.track(key(), TRACKER);
    service.track(key(), TRACKER);
    service.untrack(key(), TRACKER);
    service.untrack(key(), TRACKER);

    assertThat(contractSupplier.prefetchedKeys(), not(hasItem(key())));
  }

  @Test
  public void serviceNotificationIsIdempotent() {
    service
        .track(key(), TRACKER)
        .track(key(), TRACKER)
        .track(key(), TRACKER);

    assertThat(service.trackedApis(), hasItem(key()));
    assertThat(service.trackedApis(), not(hasItem(anotherKey())));
    assertThat(mockLogger.lines(), hasSize(3));
    assertThat(mockLogger.lines().get(0), is(trackerAddedDebugLine(key())));
    assertThat(mockLogger.lines().get(1), is(trackerAddedDebugLine(key())));
    assertThat(mockLogger.lines().get(2), is(trackerAddedDebugLine(key())));
  }

  @Test
  public void shouldNotFetchContractsIfAPIHasBeenRemoved() throws UnknownAPIException {
    service
        .track(key(), TRACKER)
        .track(anotherKey(), TRACKER)
        .untrack(key(), TRACKER);

    assertThat(service.trackedApis(), hasItem(anotherKey()));
    assertThat(service.trackedApis(), not(hasItem(key())));
    assertThat(mockLogger.lines(), hasSize(3));
    assertThat(mockLogger.lines().get(0), is(trackerAddedDebugLine(key())));
    assertThat(mockLogger.lines().get(1), is(trackerAddedDebugLine(anotherKey())));
    assertThat(mockLogger.lines().get(2), is(trackerRemovedDebugLine()));
  }

  @Test
  public void eachTrackMustBeUnTracked() {
    service
        .track(key(), TRACKER)
        .track(key(), TRACKER)
        .untrack(key(), TRACKER)
        .untrack(key(), "another name just for description");

    assertThat(service.trackedApis(), not(hasItem(key())));
  }

  @Test
  public void reTracking() {
    service
        .track(key(), TRACKER)
        .untrack(key(), TRACKER)
        .track(key(), TRACKER)
        .untrack(key(), "another name just for description");

    assertThat(service.trackedApis(), not(hasItem(key())));
  }

  @Test
  public void noApiSupplierRaisesException() throws UnknownAPIException {
    thrown.expect(UnknownAPIException.class);
    thrown.expectMessage("Cannot lookup " + key() + " as there is no ApiContractsSupplier");
    new ContractServiceImplementation().contracts(key());
  }

  @Test
  public void untrackNotTrackedApis() {
    service.untrack(key(), TRACKER);

    assertThat(service.trackedApis(), is(asList()));
    assertThat(mockLogger.lines(), hasSize(1));
    assertThat(mockLogger.lines().get(0), is(new ErrorLine("Trying to untrack Api: {} that hasn't been tracked.", key())));
  }

  private Client unknownClient() {
    return Client.builder().withId("inexistentClient").withSecret("inexistentSecret").withName("inexistantName").build();
  }

  private List<Contract> allContracts() {
    return asList(contract(), anotherContract(), aThirdContract());
  }

  private List<Contract> noContracts() {
    return new ArrayList<>();
  }

  private ApiKey key() {
    return API_KEY;
  }

  private ApiKey anotherKey() {
    return API_KEY_2;
  }

  private void assertClientNotValid(ApiContracts apiContracts, Client client) {
    try {
      apiContracts.validate(client.id(), client.secret());
      fail(client + " should not be validated!");
    } catch (ForbiddenClientException e) {
      assertEquals("Invalid client id or secret", e.getMessage());
    }
  }

  private void expectedUnknownAPI() {
    thrown.expect(UnknownAPIException.class);
    thrown.expectMessage("ContractService is not tracking " + key() + ".");
  }

  private Client client() {
    return Client.builder().withId("id").withSecret("secret").withName("some name").build();
  }

  private Client anotherClient() {
    return Client.builder().withId("another-id").withSecret("secret").withName("some name").build();
  }

  private Client aThirdClient() {
    return Client.builder().withId("third-id").withSecret("someSecretHeHasToKeep =o").withName("some name").build();
  }

  private Contract contract() {
    return Contract.builder().withClient(client()).withSla(new Sla(1)).build();
  }

  private Contract anotherContract() {
    return Contract.builder()
        .withClient(anotherClient())
        .withSla(new Sla(2, new SingleTier(3, 2L)))
        .build();
  }

  private Contract aThirdContract() {
    return Contract.builder().withClient(aThirdClient()).withSla(new Sla(3)).build();
  }

  private DebugLine trackerAddedDebugLine(ApiKey key) {
    return new DebugLine("API {} expects contracts as it is used by {}.", key.id(), TRACKER);
  }

  private DebugLine trackerRemovedDebugLine() {
    return new DebugLine("Tracker {} does not need contracts for API {} anymore.", TRACKER, key().id());
  }

}
