/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.analytics;

import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP_2;
import static com.mulesoft.anypoint.tests.http.ApacheHttpRequest.request;
import static com.mulesoft.anypoint.tests.infrastructure.model.FakeApiModel.fakeModel;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestApplication;
import static com.mulesoft.mule.runtime.gw.api.analytics.AnalyticsStatus.Status.DISABLED;
import static com.mulesoft.mule.runtime.gw.api.analytics.AnalyticsStatus.Status.ENABLED;
import static com.mulesoft.mule.runtime.gw.api.analytics.GatewayAnalytics.GW_ANALYTICS_REGISTRY_KEY;
import static com.mulesoft.mule.runtime.gw.api.config.PlatformClientConfiguration.POLL_APIS_FREQ;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;

import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tck.junit4.rule.SystemProperty;

import com.mulesoft.anypoint.tests.http.HttpRequest;
import com.mulesoft.anypoint.tita.environment.api.artifact.Artifact;
import com.mulesoft.mule.runtime.gw.api.analytics.AnalyticsStatus;
import com.mulesoft.mule.runtime.gw.api.analytics.AnalyticsStatus.Status;
import com.mulesoft.mule.runtime.gw.api.analytics.GatewayAnalytics;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;

import java.util.Optional;

import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.RuleChain;

public class AnalyticsApiLifecycleTestCase extends AbstractAnalyticsTestCase {

  private static final ApiKey EMPTY_APP2_API_KEY = new ApiKey(6L);
  private static final ApiKey NOT_FOUND_API_KEY = new ApiKey(100L);
  private static final String APP_4 = "app4";

  private static DynamicPort httpPortApp2 = new DynamicPort("portApp2");
  private static DynamicPort httpPortApp4 = new DynamicPort("portApp4");

  private static Artifact application2 = buildTestApplication(APP_2, "mule-config-http-app2.xml");

  @ClassRule
  public static RuleChain chain = RuleChain
      .outerRule(new SystemProperty(POLL_APIS_FREQ, "8"))
      .around(httpPortApp2)
      .around(httpPortApp4)
      .around(getChain(true));

  private HttpRequest emptyRequestApp2 = request(httpPortApp2.getNumber(), "/api/empty");

  @BeforeClass
  public static void beforeClass() {
    fakeModel().createApi(EMPTY_APP2_API_KEY.id());
    installation.getServer().deployApplications(application2);
  }

  @Before
  public void setUp() {
    installation.getServer().assertApiTracked(EMPTY_APP2_API_KEY);
  }

  @After
  public void tearDown() {
    fakeModel().clearEvents();
  }

  @Test
  public void appWithTrackedApi() {
    GatewayAnalytics gatewayAnalytics = getAnalyticsApi().get();

    AnalyticsStatus status = gatewayAnalytics.status(application.getName(), "empty");

    assertStatus(status, application.getName(), "empty", ENABLED, EMPTY_API_KEY);
    eventGenerated(emptyRequest);
  }

  @Test
  public void appWithApiNotFound() throws InterruptedException {
    Artifact applicationApiNotFound = buildTestApplication(APP_4, "mule-config-api-not-found.xml");
    installation.getServer().deployApplications(applicationApiNotFound);
    GatewayAnalytics gatewayAnalytics = getAnalyticsApi().get();

    AnalyticsStatus status = gatewayAnalytics.status(applicationApiNotFound.getName(), "empty");

    assertStatus(status, applicationApiNotFound.getName(), "empty", DISABLED, NOT_FOUND_API_KEY);
    noEventsGenerated(request(httpPortApp4, "/api/empty"));
  }

  @Test
  public void appWithApiWhichIsDeletedDuringRuntime() throws InterruptedException {
    GatewayAnalytics gatewayAnalytics = getAnalyticsApi().get();

    AnalyticsStatus status = gatewayAnalytics.status(application2.getName(), "empty");

    assertStatus(status, application2.getName(), "empty", ENABLED, EMPTY_APP2_API_KEY);

    eventGenerated(emptyRequestApp2);

    fakeModel().removeApi(EMPTY_APP2_API_KEY.id());

    probe(() -> {
      AnalyticsStatus updatedStatus = gatewayAnalytics.status(application2.getName(), "empty");
      assertStatus(updatedStatus, application2.getName(), "empty", DISABLED, EMPTY_APP2_API_KEY);
    });
    noEventsGenerated(emptyRequestApp2);
  }

  @Test
  public void appWithFlowWithoutApi() throws InterruptedException {
    GatewayAnalytics gatewayAnalytics = getAnalyticsApi().get();

    AnalyticsStatus status = gatewayAnalytics.status(application.getName(), "no-api");

    assertStatus(status, application.getName(), "no-api", DISABLED, null);
    noEventsGenerated(noApiRequest);
  }

  private void assertStatus(AnalyticsStatus analyticsStatus, String appName, String flowName, Status status, ApiKey apiKey) {
    assertThat(analyticsStatus.status(), is(status));
    assertThat(analyticsStatus.apiKey(), is(apiKey));
    assertThat(analyticsStatus.appName(), is(appName));
    assertThat(analyticsStatus.flowName(), is(flowName));
  }

  private void eventGenerated(HttpRequest httpRequest) {
    assertEquals(201, httpRequest.get().statusCode());

    probe(() -> assertThat(fakeModel().getEvents(), hasSize(1)));
    fakeModel().clearEvents();
  }

  private void noEventsGenerated(HttpRequest httpRequest) throws InterruptedException {
    assertEquals(201, httpRequest.get().statusCode());

    // TODO: AGW-1499: Remove sleep
    Thread.sleep(3000);

    assertThat(fakeModel().getEvents(), empty());
  }

  private Optional<GatewayAnalytics> getAnalyticsApi() {
    return installation.getServer().findApplication(application.getName()).getRegistry().lookupByName(GW_ANALYTICS_REGISTRY_KEY);
  }

}
