/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.analytics;

import static com.google.common.collect.ImmutableMap.of;
import static com.mulesoft.anypoint.tests.infrastructure.model.FakeApiModel.fakeModel;
import static org.hamcrest.Matchers.hasSize;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;
import static org.junit.rules.RuleChain.outerRule;

import org.mule.tck.junit4.rule.SystemProperty;

import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.factory.PolicyFactory;
import com.mulesoft.mule.runtime.gw.policies.service.PolicyDeploymentService;

import java.util.Map;
import java.util.stream.IntStream;

import org.junit.After;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.RuleChain;

public class AnalyticsCacheCapacityTestCase extends AbstractAnalyticsTestCase {

  private static final String POLICY_ID = "1";
  private static final String POLICY_ID_2 = "2";

  // Make sure that push is done after all requests in the tests are completed
  private static final Integer FREQ = 10;

  private static final Integer MAX_CAPACITY = 7;
  private static final Integer VIOLATIONS_MAX_CAPACITY = 5;
  private static final Integer VIOLATIONS_THRESHOLD = 3;

  private static SystemProperty cacheCapacity =
      new SystemProperty("anypoint.platform.analytics_cache_capacity", MAX_CAPACITY.toString());

  private static SystemProperty violationsCacheCapacity =
      new SystemProperty("anypoint.platform.analytics_policy_violations_cache_capacity", VIOLATIONS_MAX_CAPACITY.toString());

  private static SystemProperty violationsThresholdCacheCapacity =
      new SystemProperty("anypoint.platform.analytics_policy_violations_threshold", VIOLATIONS_THRESHOLD.toString());

  @ClassRule
  public static RuleChain chain = outerRule(cacheCapacity)
      .around(violationsCacheCapacity)
      .around(violationsThresholdCacheCapacity)
      .around(getChain(true, FREQ));

  private PolicyDeploymentService policyDeploymentService;

  @Before
  public void setUp() {
    policyDeploymentService = installation.getServer().getPolicyDeploymentService();
  }

  @After
  public void tearDown() {
    fakeModel().clearEvents();
    installation.removePoliciesAndContext();
  }

  @Test
  public void regularCacheCapacityIsHonored() {
    assertFirstPushCycleCompleted();

    IntStream.range(0, 10).forEach(value -> assertEquals(201, emptyRequest.get().statusCode()));

    probe(() -> assertThat(fakeModel().getEvents(), hasSize(MAX_CAPACITY)));
  }

  @Test
  public void violationsThresholdCapacityIsHonored() {
    int statusCode = 419;
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, ERROR_BEFORE_FLOW_TEMPLATE_KEY, EMPTY_API_KEY, null, 1,
                             new PolicyConfiguration(statusCode(statusCode)));
    policyDeploymentService.newPolicy(policyDefinition);

    assertFirstPushCycleCompleted();

    IntStream.range(0, 10).forEach(value -> assertEquals(statusCode, emptyRequest.get().statusCode()));

    probe(() -> assertThat(fakeModel().getEvents(), hasSize(VIOLATIONS_THRESHOLD)));
  }

  @Test
  public void violationsCapacityIsHonored() {
    int statusCode = 419;
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, ERROR_BEFORE_FLOW_TEMPLATE_KEY, EMPTY_API_KEY, null, 1,
                             new PolicyConfiguration(statusCode(statusCode)));
    PolicyDefinition policyDefinition2 =
        new PolicyDefinition(POLICY_ID_2, ERROR_BEFORE_FLOW_TEMPLATE_KEY, ERROR_API_KEY, null, 1,
                             new PolicyConfiguration(statusCode(statusCode)));
    policyDeploymentService.newPolicy(policyDefinition);
    policyDeploymentService.newPolicy(policyDefinition2);

    assertFirstPushCycleCompleted();

    IntStream.range(0, 5).forEach(value -> assertEquals(statusCode, emptyRequest.get().statusCode()));
    IntStream.range(0, 5).forEach(value -> assertEquals(statusCode, errorRequest.get().statusCode()));

    probe(() -> assertThat(fakeModel().getEvents(), hasSize(VIOLATIONS_MAX_CAPACITY)));
  }

  private void assertFirstPushCycleCompleted() {
    emptyRequest.get().statusCode();
    probe(() -> assertThat("Event was not present", fakeModel().getEvents(), hasSize(1)));

    fakeModel().clearEvents();
  }

  private Map<String, Object> statusCode(int statusCode) {
    return of("statusCode", statusCode);
  }

}
