/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.analytics;

import static com.google.common.collect.ImmutableMap.of;
import static com.mulesoft.anypoint.analytics.asserter.AnalyticsEventAsserter.asserter;
import static com.mulesoft.anypoint.tests.infrastructure.model.FakeApiModel.fakeModel;
import static com.mulesoft.mule.runtime.gw.analytics.AnalyticsCoreExtension.ANALYTICS_POLICY_VIOLATION_QUEUE;
import static com.mulesoft.mule.runtime.gw.analytics.AnalyticsCoreExtension.ANALYTICS_QUEUE;
import static javax.ws.rs.HttpMethod.GET;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.junit.Assert.assertThat;

import org.mule.tck.junit4.rule.SystemProperty;

import com.mulesoft.anypoint.tests.infrastructure.model.FakePolicyViolation;
import com.mulesoft.anypoint.tests.infrastructure.model.FakePolicyViolationOutcome;
import com.mulesoft.mule.runtime.gw.api.analytics.AnalyticsHttpEvent;
import com.mulesoft.mule.runtime.gw.api.analytics.PolicyViolationOutcome;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;

import java.util.Queue;

import org.junit.After;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.RuleChain;

public class AnalyticsMultipleConsumersTestCase extends AbstractAnalyticsTestCase {

  private static final String POLICY_ID = "1";

  @ClassRule
  public static RuleChain chain =
      RuleChain
          .outerRule(new SystemProperty("anypoint.platform.analytics_multiple_consumers_enabled", "true"))
          .around(getChain(true));

  @After
  public void tearDown() {
    installation.removePoliciesAndContext();
    getQueue(ANALYTICS_QUEUE).clear();
    getQueue(ANALYTICS_POLICY_VIOLATION_QUEUE).clear();
    fakeModel().clearEvents();
  }

  @Test
  public void analyticsEventAvailableInBothRegistryAndCloud() {
    Queue<AnalyticsHttpEvent> queue = getQueue(ANALYTICS_QUEUE);
    Queue<AnalyticsHttpEvent> violationsQueue = getQueue(ANALYTICS_POLICY_VIOLATION_QUEUE);

    assertThat(emptyRequest.withHeader("User-Agent", USER_AGENT).get().statusCode(), is(201));

    probe(() -> {

      assertThat(queue.size(), is(1));
      assertThat(violationsQueue.isEmpty(), is(true));

      assertThat(fakeModel().getEvents(), hasSize(1));

      asserter()
          .methodIs(GET)
          .pathIs("/api/empty")
          .statusCodeIs(201)
          .execute(fakeModel().getFirstEvent());
    });
  }

  @Test
  public void violationsDetectedWhenAnalyticsThroughAgentAndCloud() {
    int statusCode = 501;
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, ERROR_BEFORE_FLOW_TEMPLATE_KEY, EMPTY_API_KEY, null, 1,
                             new PolicyConfiguration(of("statusCode", statusCode)));
    installation.getServer().getPolicyDeploymentService()
        .newPolicy(policyDefinition);

    assertThat(emptyRequest.withHeader("User-Agent", USER_AGENT).get().statusCode(), is(statusCode));

    Queue<AnalyticsHttpEvent> queue = getQueue(ANALYTICS_QUEUE);
    Queue<AnalyticsHttpEvent> violationsQueue = getQueue(ANALYTICS_POLICY_VIOLATION_QUEUE);

    probe(() -> {
      assertThat(queue.isEmpty(), is(true));
      assertThat(violationsQueue.size(), is(1));

      AnalyticsHttpEvent event = violationsQueue.peek();

      assertThat("Analytics event is null", event, notNullValue());
      assertThat("Policy violation is null", event.getPolicyViolation(), notNullValue());
      assertThat(event.getPolicyViolation().getOutcome(), is(PolicyViolationOutcome.VIOLATION));
      assertThat(event.getPolicyViolation().getPolicyId(), is("1"));
      assertThat(event.getPolicyViolation().getPolicyName(), is(policyDefinition.getName()));

      assertThat(fakeModel().getEvents(), hasSize(1));

      FakePolicyViolation policyViolation = FakePolicyViolation.create(POLICY_ID, null, FakePolicyViolationOutcome.VIOLATION);

      asserter()
          .methodIs(GET)
          .pathIs("/api/empty")
          .statusCodeIs(statusCode)
          .violationIs(policyViolation)
          .execute(fakeModel().getFirstEvent());
    });
  }

  private Queue<AnalyticsHttpEvent> getQueue(String queueName) {
    return installation.getServer().findApplication(application.getName()).getRegistry().<Queue>lookupByName(queueName)
        .get();
  }

}
