/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.analytics;

import static com.google.common.collect.ImmutableMap.of;
import static com.mulesoft.anypoint.analytics.asserter.AnalyticsEventAsserter.asserter;
import static com.mulesoft.anypoint.tests.infrastructure.model.FakeApiModel.fakeModel;
import static com.mulesoft.anypoint.tests.infrastructure.model.FakePolicyViolationOutcome.VIOLATION;
import static java.util.Collections.emptyMap;
import static javax.ws.rs.HttpMethod.GET;
import static junit.framework.Assert.assertEquals;
import static org.hamcrest.Matchers.hasSize;
import static org.junit.Assert.assertThat;

import com.mulesoft.anypoint.tests.infrastructure.model.FakePolicyViolation;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;

import org.junit.After;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.RuleChain;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.runners.Parameterized.Parameters;

@RunWith(Parameterized.class)
public class AnalyticsOnPremTestCase extends AbstractAnalyticsTestCase {

  private static final String POLICY_ID = "1";
  private static final String ON_PREM = "anypoint.platform.on_prem";
  private static final String CLIENT_ID = "clientId";
  private static final String CLIENT_NAME = "clientName";

  @ClassRule
  public static RuleChain chain = RuleChain.outerRule(getChain(true));

  private boolean isOnPrem;

  public AnalyticsOnPremTestCase(boolean isOnPrem) {
    this.isOnPrem = isOnPrem;
  }

  @Parameters(name = "OnPrem: {0}")
  public static Object[][] setIsOnPrem() {
    return new Object[][] {
        {false},
        {true}
    };
  }

  @Before
  public void setUp() {
    System.setProperty(ON_PREM, String.valueOf(isOnPrem));
  }

  @After
  public void tearDown() {
    installation.removePoliciesAndContext();
    fakeModel().clearEvents();
    System.clearProperty(ON_PREM);
  }

  @Test
  public void apiAndpolicyNameVisibleForOnPrem() {
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, ERROR_BEFORE_FLOW_TEMPLATE_KEY, EMPTY_API_KEY, null, 1,
                             new PolicyConfiguration(of("statusCode", 501)));
    installation.getServer().getPolicyDeploymentService()
        .newPolicy(policyDefinition);

    assertEquals(501, emptyRequest.withHeader("User-Agent", USER_AGENT).get().statusCode());

    probe(() -> {
      assertThat(fakeModel().getEvents(), hasSize(1));

      FakePolicyViolation expectedViolation = isOnPrem
          ? FakePolicyViolation.create(POLICY_ID, policyDefinition.getName(), VIOLATION)
          : FakePolicyViolation.create(POLICY_ID, null, VIOLATION);

      asserter()
          .methodIs(GET)
          .pathIs("/api/empty")
          .statusCodeIs(501)
          .apiNameIs(isOnPrem ? EMPTY_EXCHANGE_NAME : null)
          .apiVersionIs(isOnPrem ? EMPTY_API_VERSION : null)
          .instanceNameIs(isOnPrem ? EMPTY_INSTANCE_NAME : null)
          .violationIs(expectedViolation)
          .execute(fakeModel().getFirstEvent());
    });
  }

  @Test
  public void applicationNameVisibleForOnPrem() {
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, WITH_CLIENT_TEMPLATE_KEY, EMPTY_API_KEY, null, 1, new PolicyConfiguration(emptyMap()));
    installation.getServer().deployPolicy(policyDefinition);

    assertEquals(201, emptyRequest.withHeader("User-Agent", USER_AGENT).get().statusCode());

    probe(() -> {
      assertThat(fakeModel().getEvents(), hasSize(1));

      asserter()
          .methodIs(GET)
          .pathIs("/api/empty")
          .statusCodeIs(201)
          .applicationIdIs(CLIENT_ID)
          .applicationNameIs(isOnPrem ? CLIENT_NAME : null)
          .apiNameIs(isOnPrem ? EMPTY_EXCHANGE_NAME : null)
          .apiVersionIs(isOnPrem ? EMPTY_API_VERSION : null)
          .instanceNameIs(isOnPrem ? EMPTY_INSTANCE_NAME : null)
          .execute(fakeModel().getFirstEvent());
    });
  }

}
