/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.analytics;

import static com.mulesoft.anypoint.analytics.asserter.AnalyticsEventAsserter.asserter;
import static com.mulesoft.anypoint.tests.infrastructure.model.FakeApiModel.fakeModel;
import static javax.ws.rs.HttpMethod.GET;
import static javax.ws.rs.HttpMethod.POST;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_LENGTH;

import com.mulesoft.anypoint.tests.http.HttpResponse;
import com.mulesoft.anypoint.tests.http.SimpleHttpServerResponse;

import org.junit.After;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.RuleChain;

public class AnalyticsPayloadSizeTestCase extends AbstractAnalyticsTestCase {

  private static final String BACKEND_PAYLOAD = "Another";
  public static final String REQUEST_PAYLOAD = "Payload";

  @ClassRule
  public static RuleChain chain = RuleChain.outerRule(getChain(true));

  @After
  public void tearDown() {
    fakeModel().clearEvents();
    installation.removePoliciesAndContext();
  }

  @Test
  public void requestAndResponseBytes() {
    HttpResponse response = responseRequest.withHeader("User-Agent", USER_AGENT).withPayload(REQUEST_PAYLOAD).post();

    assertThat(response.statusCode(), is(200));
    assertThat(response.header(CONTENT_LENGTH), is(String.valueOf(RESPONSE_PAYLOAD.length())));

    probe(() -> {
      assertThat(fakeModel().getEvents(), hasSize(1));

      asserter()
          .methodIs(POST)
          .pathIs("/api/response")
          .statusCodeIs(200)
          .responseBytesAre(RESPONSE_PAYLOAD.length())
          .requestBytesAre(REQUEST_PAYLOAD.length())
          .execute(fakeModel().getFirstEvent());
    });
  }

  @Test
  public void requestGetMethodReportedAsZeroBytes() {
    assertEquals(200, responseRequest.withHeader("User-Agent", USER_AGENT).get().statusCode());

    probe(() -> {
      assertThat(fakeModel().getEvents(), hasSize(1));

      asserter()
          .methodIs(GET)
          .pathIs("/api/response")
          .statusCodeIs(200)
          .requestBytesAre(0)
          .execute(fakeModel().getFirstEvent());
    });
  }

  @Test
  public void requestPostChunkedReportedAsUndefinedBytes() {
    HttpResponse response = responseRequest.withHeader("User-Agent", USER_AGENT)
        .withPayload(REQUEST_PAYLOAD)
        .chunked()
        .post();

    assertThat(response.statusCode(), is(200));

    probe(() -> {
      assertThat(fakeModel().getEvents(), hasSize(1));

      asserter()
          .methodIs(POST)
          .pathIs("/api/response")
          .statusCodeIs(200)
          .requestBytesAre(-1)
          .execute(fakeModel().getFirstEvent());
    });
  }

  @Test
  public void explicitResponseHeaderContentLengthSetInFlowIsHonored() {
    HttpResponse response = explicitClRequest.withHeader("User-Agent", USER_AGENT).get();

    assertThat(response.statusCode(), is(200));
    assertThat(response.header(CONTENT_LENGTH), is("15"));

    probe(() -> {
      assertThat(fakeModel().getEvents(), hasSize(1));

      asserter()
          .methodIs(GET)
          .pathIs("/api/explicit-cl")
          .statusCodeIs(200)
          .responseBytesAre(2)
          .execute(fakeModel().getFirstEvent());
    });
  }

  @Test
  public void backendReturnsContentLength() {
    SimpleHttpServerResponse backendResponse = SimpleHttpServerResponse.builder().body(BACKEND_PAYLOAD).build();
    backendServer.getHttpServer().setResponse(backendResponse);

    HttpResponse response = proxyRequest.withHeader("User-Agent", USER_AGENT).get();

    assertThat(response.statusCode(), is(200));
    assertThat(response.header(CONTENT_LENGTH), is(String.valueOf(BACKEND_PAYLOAD.length())));

    probe(() -> {
      assertThat(fakeModel().getEvents(), hasSize(1));

      asserter()
          .methodIs(GET)
          .pathIs("/api/proxy")
          .statusCodeIs(200)
          .responseBytesAre(BACKEND_PAYLOAD.length())
          .execute(fakeModel().getFirstEvent());
    });
  }

  @Test
  public void backendReturnsChunked() {
    SimpleHttpServerResponse backendResponse = SimpleHttpServerResponse.builder().body(BACKEND_PAYLOAD).chunked(true).build();
    backendServer.getHttpServer().setResponse(backendResponse);

    HttpResponse response = proxyRequest.withHeader("User-Agent", USER_AGENT).get();

    assertThat(response.statusCode(), is(200));
    assertThat(response.header(CONTENT_LENGTH), nullValue());

    probe(() -> {
      assertThat(fakeModel().getEvents(), hasSize(1));

      asserter()
          .methodIs(GET)
          .pathIs("/api/proxy")
          .statusCodeIs(200)
          .responseBytesAre(-1)
          .execute(fakeModel().getFirstEvent());
    });
  }

}
