/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.analytics;

import static com.mulesoft.anypoint.analytics.asserter.AnalyticsEventAsserter.asserter;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.anypoint.tests.infrastructure.model.FakeApiModel.fakeModel;
import static com.mulesoft.anypoint.tests.infrastructure.model.FakePolicyViolationOutcome.VIOLATION;
import static java.util.Collections.emptyMap;
import static javax.ws.rs.HttpMethod.GET;
import static org.hamcrest.Matchers.hasSize;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;

import com.mulesoft.anypoint.tests.infrastructure.model.FakePolicyViolation;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.OfflinePolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.factory.PolicyFactory;
import com.mulesoft.mule.runtime.gw.policies.service.PolicyDeploymentService;

import com.google.common.collect.ImmutableMap;

import java.util.Map;

import org.junit.After;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.rules.RuleChain;

public class AnalyticsPolicyViolationTestCase extends AbstractAnalyticsTestCase {

  private static final String POLICY_ID = "1";
  private static final String POLICY_ID_2 = "2";

  @ClassRule
  public static RuleChain chain = getChain(true);

  private PolicyDeploymentService policyDeploymentService;

  @Before
  public void setUp() {
    policyDeploymentService = installation.getServer().getPolicyDeploymentService();
  }

  @After
  public void tearDown() {
    fakeModel().clearEvents();
    installation.removePoliciesAndContext();
  }

  @Test
  public void errorInPolicyBeforeFlowIsDetectedAsViolation() {
    int statusCode = 501;
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, ERROR_BEFORE_FLOW_TEMPLATE_KEY, EMPTY_API_KEY, null, 1,
                             new PolicyConfiguration(statusCode(statusCode)));
    policyDeploymentService.newPolicy(policyDefinition);

    assertEquals(statusCode, emptyRequest.withHeader("User-Agent", USER_AGENT).get().statusCode());

    probe(() -> {
      assertThat("Event was not present", fakeModel().getEvents(), hasSize(1));

      FakePolicyViolation expectedViolation = FakePolicyViolation.create(POLICY_ID, null, VIOLATION);

      asserter()
          .methodIs(GET)
          .pathIs("/api/empty")
          .statusCodeIs(statusCode)
          .violationIs(expectedViolation)
          .execute(fakeModel().getFirstEvent());
    });
  }

  @Test
  public void errorInPolicyAfterFlowIsDetectedAsViolation() {
    int statusCode = 501;
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, ERROR_AFTER_FLOW_TEMPLATE_KEY, EMPTY_API_KEY, null, 1,
                             new PolicyConfiguration(statusCode(statusCode)));
    policyDeploymentService.newPolicy(policyDefinition);

    assertEquals(statusCode, emptyRequest.withHeader("User-Agent", USER_AGENT).get().statusCode());

    probe(() -> {
      assertThat(fakeModel().getEvents(), hasSize(1));

      FakePolicyViolation expectedViolation = FakePolicyViolation.create(POLICY_ID, null, VIOLATION);

      asserter()
          .methodIs(GET)
          .pathIs("/api/empty")
          .statusCodeIs(statusCode)
          .violationIs(expectedViolation)
          .execute(fakeModel().getFirstEvent());
    });
  }

  @Test
  public void errorInPolicyWithErrorPropagateIsDetectedAsViolation() {
    int statusCode = 501;
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, ERROR_PROPAGATE_TEMPLATE_KEY, EMPTY_API_KEY, null, 1,
                             new PolicyConfiguration(statusCode(statusCode)));
    policyDeploymentService.newPolicy(policyDefinition);

    assertEquals(statusCode, emptyRequest.withHeader("User-Agent", USER_AGENT).get().statusCode());

    probe(() -> {
      assertThat(fakeModel().getEvents(), hasSize(1));

      FakePolicyViolation expectedViolation = FakePolicyViolation.create(POLICY_ID, null, VIOLATION);

      asserter()
          .methodIs(GET)
          .pathIs("/api/empty")
          .statusCodeIs(statusCode)
          .violationIs(expectedViolation)
          .execute(fakeModel().getFirstEvent());
    });
  }

  @Test
  public void errorInPolicyWithNoErrorHandlerNotDetectedAsViolation() {
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, NO_ERROR_HANDLER_TEMPLATE_KEY, EMPTY_API_KEY, null, 1, new PolicyConfiguration(null));
    policyDeploymentService.newPolicy(policyDefinition);

    assertEquals(201, emptyRequest.withHeader("User-Agent", USER_AGENT).get().statusCode());

    probe(() -> {
      assertThat(fakeModel().getEvents(), hasSize(1));

      asserter()
          .methodIs(GET)
          .pathIs("/api/empty")
          .statusCodeIs(201)
          .noViolation()
          .execute(fakeModel().getFirstEvent());
    });
  }

  @Test
  public void errorInFirstPolicyWithMultiplePolicies() {
    int statusCode = 501;
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, ERROR_AFTER_FLOW_TEMPLATE_KEY, EMPTY_API_KEY, null, 1,
                             new PolicyConfiguration(statusCode(statusCode)));
    PolicyDefinition policyDefinition2 =
        new PolicyDefinition(POLICY_ID_2, POLICY_TEMPLATE_KEY, EMPTY_API_KEY, null, 2, emptyMap());
    policyDeploymentService.newPolicy(policyDefinition);
    policyDeploymentService.newPolicy(policyDefinition2);

    assertEquals(statusCode, emptyRequest.withHeader("User-Agent", USER_AGENT).get().statusCode());

    probe(() -> {
      assertThat(fakeModel().getEvents(), hasSize(1));

      FakePolicyViolation expectedViolation = FakePolicyViolation.create(policyDefinition.getId(), null, VIOLATION);

      asserter()
          .methodIs(GET)
          .pathIs("/api/empty")
          .statusCodeIs(statusCode)
          .violationIs(expectedViolation)
          .execute(fakeModel().getFirstEvent());
    });
  }

  @Test
  public void errorInSecondPolicyWithMultiplePolicies() {
    int statusCode = 501;
    PolicyDefinition policyDefinition = new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, EMPTY_API_KEY, null, 1, emptyMap());
    PolicyDefinition policyDefinition2 =
        new PolicyDefinition(POLICY_ID_2, ERROR_AFTER_FLOW_TEMPLATE_KEY, EMPTY_API_KEY, null, 2,
                             new PolicyConfiguration(statusCode(statusCode)));
    policyDeploymentService.newPolicy(policyDefinition);
    policyDeploymentService.newPolicy(policyDefinition2);

    assertEquals(statusCode, emptyRequest.withHeader("User-Agent", USER_AGENT).get().statusCode());

    probe(() -> {
      assertThat(fakeModel().getEvents(), hasSize(1));

      FakePolicyViolation expectedViolation = FakePolicyViolation.create(policyDefinition2.getId(), null, VIOLATION);

      asserter()
          .methodIs(GET)
          .pathIs("/api/empty")
          .statusCodeIs(statusCode)
          .violationIs(expectedViolation)
          .execute(fakeModel().getFirstEvent());
    });
  }

  @Test
  public void errorInFlowNotDetectedAsViolation() {
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, ERROR_API_KEY, null, 1, new PolicyConfiguration(emptyMap()));
    policyDeploymentService.newPolicy(policyDefinition);

    assertEquals(502, errorRequest.withHeader("User-Agent", USER_AGENT).get().statusCode());

    probe(() -> {
      assertThat(fakeModel().getEvents(), hasSize(1));

      asserter()
          .methodIs(GET)
          .pathIs("/api/error")
          .statusCodeIs(502)
          .noViolation()
          .execute(fakeModel().getFirstEvent());
    });
  }

  @Test
  public void errorInFlowWithPolicyWithErrorHandlerSettingStatusCodeAbove400NotDetectedAsViolation() {
    int policyStatusCode = 501;
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, ERROR_AFTER_FLOW_TEMPLATE_KEY, ERROR_API_KEY, null, 1,
                             new PolicyConfiguration(statusCode(policyStatusCode)));
    policyDeploymentService.newPolicy(policyDefinition);

    assertEquals(policyStatusCode, errorRequest.withHeader("User-Agent", USER_AGENT).get().statusCode());

    probe(() -> {
      assertThat(fakeModel().getEvents(), hasSize(1));

      asserter()
          .methodIs(GET)
          .pathIs("/api/error")
          .statusCodeIs(policyStatusCode)
          .noViolation()
          .execute(fakeModel().getFirstEvent());
    });
  }

  @Test
  public void offlinePolicyViolationNotDetectedAsViolation() {
    int statusCode = 501;
    OfflinePolicyDefinition policyDefinition =
        new OfflinePolicyDefinition(POLICY_ID, ERROR_BEFORE_FLOW_TEMPLATE_KEY, EMPTY_API_KEY, null, 1,
                                    new PolicyConfiguration(statusCode(statusCode)));
    policyDeploymentService.newPolicy(policyDefinition);

    assertEquals(statusCode, emptyRequest.withHeader("User-Agent", USER_AGENT).get().statusCode());

    probe(() -> {
      assertThat(fakeModel().getEvents(), hasSize(1));

      asserter()
          .methodIs(GET)
          .pathIs("/api/empty")
          .statusCodeIs(statusCode)
          .noViolation()
          .execute(fakeModel().getFirstEvent());
    });
  }

  @Test
  public void errorInPolicyFixedInNextPolicyNotDetectedAsViolation() {
    int statusCode1 = 204;
    int statusCode2 = 501;
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, SET_STATUS_TEMPLATE_KEY, EMPTY_API_KEY, null, 1,
                             new PolicyConfiguration(statusCode(statusCode1)));
    PolicyDefinition policyDefinition2 =
        new PolicyDefinition(POLICY_ID_2, ERROR_BEFORE_FLOW_TEMPLATE_KEY, EMPTY_API_KEY, null, 2,
                             new PolicyConfiguration(statusCode(statusCode2)));
    policyDeploymentService.newPolicy(policyDefinition);
    policyDeploymentService.newPolicy(policyDefinition2);

    assertEquals(statusCode1, emptyRequest.withHeader("User-Agent", USER_AGENT).get().statusCode());

    probe(() -> {
      assertThat(fakeModel().getEvents(), hasSize(1));

      asserter()
          .methodIs(GET)
          .pathIs("/api/empty")
          .statusCodeIs(statusCode1)
          .noViolation()
          .execute(fakeModel().getFirstEvent());
    });
  }

  @Test
  public void errorInPolicyFixedInNextPolicyWhenErrorPropagateNotDetectedAsViolation() {
    int statusCode1 = 204;
    int statusCode2 = 501;
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, SET_STATUS_ON_ERROR_TEMPLATE_KEY, EMPTY_API_KEY, null, 1,
                             new PolicyConfiguration(statusCode(statusCode1)));
    PolicyDefinition policyDefinition2 =
        new PolicyDefinition(POLICY_ID_2, ERROR_PROPAGATE_TEMPLATE_KEY, EMPTY_API_KEY, null, 2,
                             new PolicyConfiguration(statusCode(statusCode2)));
    policyDeploymentService.newPolicy(policyDefinition);
    policyDeploymentService.newPolicy(policyDefinition2);

    assertEquals(statusCode1, emptyRequest.withHeader("User-Agent", USER_AGENT).get().statusCode());

    probe(() -> {
      assertThat(fakeModel().getEvents(), hasSize(1));

      asserter()
          .methodIs(GET)
          .pathIs("/api/empty")
          .statusCodeIs(statusCode1)
          .noViolation()
          .execute(fakeModel().getFirstEvent());
    });
  }

  private Map<String, Object> statusCode(int statusCode) {
    return ImmutableMap.of("statusCode", statusCode);
  }

}
