/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.analytics;

import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP_3;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID_3;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.anypoint.tests.http.ApacheHttpRequest.request;
import static com.mulesoft.anypoint.tests.infrastructure.model.FakeApiModel.fakeModel;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestApplication;
import static com.mulesoft.mule.runtime.gw.analytics.servicemesh.ServiceMeshHeaderInjector.SEPARATOR;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static org.hamcrest.Matchers.anyOf;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;

import org.mule.tck.junit4.rule.DynamicPort;

import com.mulesoft.anypoint.tests.http.HttpRequest;
import com.mulesoft.anypoint.tests.http.HttpResponse;
import com.mulesoft.anypoint.tita.environment.api.artifact.Artifact;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.config.AnalyticsConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;

import com.google.common.collect.ImmutableMap;

import java.util.Arrays;
import java.util.Base64;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.RuleChain;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(Parameterized.class)
public class AnalyticsServiceMeshTestCase extends AbstractAnalyticsTestCase {

  private static final String SM_HEADER = new AnalyticsConfiguration().getServiceMeshHeader();
  private static final int NON_VIOLATION_PARTS_COUNT = 5;
  private static final int VIOLATION_PARTS_COUNT = 9;

  private static final ApiKey HDP_SVC1_APP3_API_KEY = new ApiKey(7L);
  private static final ApiKey HDP_SVC2_APP3_API_KEY = new ApiKey(8L);

  private static final String HDP_REGISTRY = "app3-0.0.1-mule-application.json";

  private static DynamicPort httpPortApp3 = new DynamicPort("portApp3");

  @ClassRule
  public static RuleChain chain = RuleChain.outerRule(httpPortApp3).around(getChain(true, 1, true));

  private static Artifact applicationHdp = buildTestApplication(APP_3, "mule-config-http-app3-hdp.xml",
                                                                singletonList("app3-hdp-config.properties"));

  private HttpRequest hdpService1Request =
      request(httpPortApp3.getNumber(), "/api/empty").withHeader("X-MULE-HDP-SERVICE", "svc1");
  private HttpRequest hdpService2Request =
      request(httpPortApp3.getNumber(), "/api/empty").withHeader("X-MULE-HDP-SERVICE", "svc2");

  private boolean deployPolicy;

  public AnalyticsServiceMeshTestCase(boolean deployPolicy) {
    this.deployPolicy = deployPolicy;
  }

  @Parameterized.Parameters
  public static Collection<Object[]> data() {
    return Arrays.asList(new Object[][] {
        {false},
        {true}
    });
  }

  @BeforeClass
  public static void beforeClass() {
    fakeModel().createApi(HDP_SVC1_APP3_API_KEY.id());
    fakeModel().createApi(HDP_SVC2_APP3_API_KEY.id());
    installation.updateHdpRegistry(HDP_REGISTRY);
    installation.getServer().deployApplications(applicationHdp);
  }

  @Before
  public void setUp() {
    installation.getServer().assertApiTracked(HDP_SVC1_APP3_API_KEY);
    installation.getServer().assertApiTracked(HDP_SVC2_APP3_API_KEY);

    if (deployPolicy) {
      PolicyDefinition emptyApiPolicySvc1 =
          new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, HDP_SVC1_APP3_API_KEY, null, 1,
                               new PolicyConfiguration(emptyMap()));
      PolicyDefinition emptyApiPolicySvc2 =
          new PolicyDefinition(POLICY_ID_2, POLICY_TEMPLATE_KEY, HDP_SVC2_APP3_API_KEY, null, 1,
                               new PolicyConfiguration(emptyMap()));

      installation.getServer().deployPolicy(emptyApiPolicySvc1);
      installation.getServer().deployPolicy(emptyApiPolicySvc2);
    }
  }

  @After
  public void tearDown() {
    fakeModel().clearEvents();
    installation.removePoliciesAndContext();
  }

  @Test
  public void nonViolationReportHeader() {
    HttpResponse httpResponse = hdpService1Request.withHeader("User-Agent", USER_AGENT).get();
    assertEquals(201, httpResponse.statusCode());
    Map<String, String> parts = splitToMap(httpResponse.header(SM_HEADER));
    assertThat(parts.keySet(), hasSize(NON_VIOLATION_PARTS_COUNT));
    assertThat(parts.get("org_id").matches("[0-9]+"), is(true));
    assertThat(parts.get("api_id"), is("99"));
    assertThat(parts.get("api_version_id"), is("7"));
    assertThat(parts.get("client_id"), is("null"));
    assertThat(parts.get("request_disposition"), is("processed"));
  }

  @Test
  public void clientIdReportHeader() {
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID_3, WITH_CLIENT_TEMPLATE_KEY, HDP_SVC1_APP3_API_KEY, null, 1,
                             new PolicyConfiguration(emptyMap()));
    installation.getServer().deployPolicy(policyDefinition);

    HttpResponse httpResponse = hdpService1Request.withHeader("User-Agent", USER_AGENT).get();
    assertEquals(201, httpResponse.statusCode());
    Map<String, String> parts = splitToMap(httpResponse.header(SM_HEADER));
    assertThat(parts.keySet(), hasSize(NON_VIOLATION_PARTS_COUNT));
    assertThat(parts.get("org_id").matches("[0-9]+"), is(true));
    assertThat(parts.get("api_id"), is("99"));
    assertThat(parts.get("api_version_id"), is("7"));
    assertThat(parts.get("client_id"), is("clientId"));
    assertThat(parts.get("request_disposition"), is("processed"));
  }

  @Test
  public void violationReportHeader() {
    PolicyDefinition errorApiPolicy =
        new PolicyDefinition(POLICY_ID_3, ERROR_PROPAGATE_TEMPLATE_KEY, HDP_SVC2_APP3_API_KEY, null, 1,
                             new PolicyConfiguration(ImmutableMap.of("statusCode", 500)));
    installation.getServer().deployPolicy(errorApiPolicy);

    HttpResponse httpResponse = hdpService2Request.withHeader("User-Agent", USER_AGENT).withHeader("client_id", "1122").get();
    assertEquals(500, httpResponse.statusCode());
    Map<String, String> parts = splitToMap(httpResponse.header(SM_HEADER));
    assertThat(parts.keySet(), hasSize(VIOLATION_PARTS_COUNT));
    assertThat(parts.get("org_id").matches("[0-9]+"), is(true));
    assertThat(parts.get("api_id"), is("99"));
    assertThat(parts.get("api_version_id"), is("8"));
    assertThat(parts.get("client_id"), is("null"));
    assertThat(parts.get("request_disposition"), is("blocked"));
    assertThat(parts.get("policy_id"), is("policyId-3"));
    assertThat(parts.get("policy_name"), is("errorPropagate-policyId-3"));
    assertThat(parts.get("category"), is("fb-loop-category"));
    assertThat(parts.get("outcome"), is("violation"));
  }

  private Map<String, String> splitToMap(String header) {
    String decoded = new String(Base64.getUrlDecoder().decode(header.getBytes()));
    String[] parts = decoded.split("\\" + SEPARATOR);
    assertThat(parts.length, anyOf(is(NON_VIOLATION_PARTS_COUNT), is(VIOLATION_PARTS_COUNT)));
    Map<String, String> partsMap = new HashMap<>();
    partsMap.put("org_id", parts[0]);
    partsMap.put("api_id", parts[1]);
    partsMap.put("api_version_id", parts[2]);
    partsMap.put("client_id", parts[3]);
    partsMap.put("request_disposition", parts[4]);
    if (parts.length == VIOLATION_PARTS_COUNT) {
      partsMap.put("policy_id", parts[5]);
      partsMap.put("policy_name", parts[6]);
      partsMap.put("category", parts[7]);
      partsMap.put("outcome", parts[8]);
    }
    return partsMap;
  }
}
