/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.analytics;

import static com.google.common.collect.ImmutableMap.of;
import static com.mulesoft.mule.runtime.gw.analytics.AnalyticsCoreExtension.ANALYTICS_POLICY_VIOLATION_QUEUE;
import static com.mulesoft.mule.runtime.gw.analytics.AnalyticsCoreExtension.ANALYTICS_QUEUE;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;

import com.mulesoft.mule.runtime.gw.api.analytics.AnalyticsHttpEvent;
import com.mulesoft.mule.runtime.gw.api.analytics.PolicyViolationOutcome;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;

import java.util.Queue;

import org.junit.After;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.RuleChain;

public class AnalyticsThroughMuleAgentTestCase extends AbstractAnalyticsTestCase {

  private static final String POLICY_ID = "1";

  @ClassRule
  public static RuleChain chain = getChain(false);

  @After
  public void tearDown() {
    installation.removePoliciesAndContext();
    getQueue(ANALYTICS_QUEUE).clear();
    getQueue(ANALYTICS_POLICY_VIOLATION_QUEUE).clear();
  }

  @Test
  public void analyticsQueueAvailableInRegistry() {
    Queue<AnalyticsHttpEvent> queue = getQueue(ANALYTICS_QUEUE);
    Queue<AnalyticsHttpEvent> violationsQueue = getQueue(ANALYTICS_POLICY_VIOLATION_QUEUE);

    assertThat(emptyRequest.withHeader("User-Agent", USER_AGENT).get().statusCode(), is(201));

    probe(() -> {
      assertNotNull(queue.peek());
      assertNull(violationsQueue.peek());
    });
  }

  @Test
  public void violationsDetectedWhenAnalyticsThroughAgent() {
    int statusCode = 501;
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, ERROR_BEFORE_FLOW_TEMPLATE_KEY, EMPTY_API_KEY, null, 1,
                             new PolicyConfiguration(of("statusCode", statusCode)));
    installation.getServer().getPolicyDeploymentService()
        .newPolicy(policyDefinition);

    assertThat(emptyRequest.get().statusCode(), is(statusCode));

    Queue<AnalyticsHttpEvent> queue = getQueue(ANALYTICS_QUEUE);
    Queue<AnalyticsHttpEvent> violationsQueue = getQueue(ANALYTICS_POLICY_VIOLATION_QUEUE);

    probe(() -> {
      assertThat(violationsQueue.size(), is(1));

      AnalyticsHttpEvent event = violationsQueue.peek();

      assertThat("Analytics event is null", event, notNullValue());
      assertThat("Policy violation is null", event.getPolicyViolation(), notNullValue());
      assertThat(event.getPolicyViolation().getOutcome(), is(PolicyViolationOutcome.VIOLATION));
      assertThat(event.getPolicyViolation().getPolicyId(), is("1"));
      assertThat(event.getPolicyViolation().getPolicyName(), is(policyDefinition.getName()));
      assertThat(queue.isEmpty(), is(true));
    });
  }

  @Test
  public void analyticsEventQueueThroughAgentContainsAppNameAndFlowName() {
    int statusCode = 501;
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, ERROR_BEFORE_FLOW_TEMPLATE_KEY, EMPTY_API_KEY, null, 1,
                             new PolicyConfiguration(of("statusCode", statusCode)));
    installation.getServer().getPolicyDeploymentService()
        .newPolicy(policyDefinition);

    assertThat(emptyRequest.get().statusCode(), is(statusCode));

    Queue<AnalyticsHttpEvent> queue = getQueue(ANALYTICS_QUEUE);
    Queue<AnalyticsHttpEvent> violationsQueue = getQueue(ANALYTICS_POLICY_VIOLATION_QUEUE);

    probe(() -> {
      assertThat(violationsQueue.size(), is(1));

      AnalyticsHttpEvent event = violationsQueue.peek();
      assertThat("Analytics event is null", event, notNullValue());
      assertThat("AppName is not correctly set", event.getMuleAppName(), is(application.getName()));
      assertThat("FlowName is not correctly set", event.getFlowName(), is("empty"));
      assertThat(queue.isEmpty(), is(true));
    });
  }

  private Queue<AnalyticsHttpEvent> getQueue(String queueName) {
    return installation.getServer().findApplication(application.getName()).getRegistry().<Queue>lookupByName(queueName)
        .get();
  }

}
