/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.analytics;

import static com.google.common.collect.ImmutableMap.of;
import static com.mulesoft.anypoint.analytics.asserter.AnalyticsEventAsserter.asserter;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID_2;
import static com.mulesoft.anypoint.tests.http.ApacheHttpRequest.request;
import static com.mulesoft.anypoint.tests.infrastructure.model.FakeApiModel.fakeModel;
import static com.mulesoft.anypoint.tests.infrastructure.model.FakePolicyViolationOutcome.VIOLATION;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestApplication;
import static javax.ws.rs.HttpMethod.GET;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;

import org.mule.tck.junit4.rule.DynamicPort;

import com.mulesoft.anypoint.tests.http.HttpRequest;
import com.mulesoft.anypoint.tests.http.HttpResponse;
import com.mulesoft.anypoint.tests.infrastructure.model.FakePolicyViolation;
import com.mulesoft.anypoint.tita.environment.api.artifact.Artifact;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;

import java.util.Arrays;
import java.util.Collection;

import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.RuleChain;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(Parameterized.class)
public class FeedbackLoopHeaderTestCase extends AbstractAnalyticsTestCase {

  private static final String FEEDBACK_LOOP_HEADER = "X-MULE-DDOS-VIOLATION";
  private static final String VIA_HEADER_NAME = "via";
  private static final String VIA_EDGE_HEADER = "EDGE/1.1 MuleSoft";
  private static final String VIA_EDGE_HEADER_ANY = "EDGE/100.99 MuleSoft";
  private static final String POLICY_ID = "1";
  private static final int STATUS_CODE = 500;
  private static final String CATEGORY = "fb-loop-category";
  private static final String CATEGORY_2 = "fb-loop-category2";
  private static final FakePolicyViolation EXPECTED_VIOLATION = FakePolicyViolation.create(POLICY_ID, null, VIOLATION);
  private static final FakePolicyViolation EXPECTED_VIOLATION2 = FakePolicyViolation.create(POLICY_ID_2, null, VIOLATION);

  private static final ApiKey EMPTY_APP2_API_KEY = new ApiKey(6L);

  private static DynamicPort httpPortApp2 = new DynamicPort("portApp2");
  private static Artifact application2 = buildTestApplication(APP_2, "mule-config-http-app2.xml");

  @ClassRule
  public static RuleChain chain = RuleChain.outerRule(httpPortApp2).around(getChain(true));

  private HttpRequest emptyRequestApp2 = request(httpPortApp2.getNumber(), "/api/empty");

  private PolicyDefinition policyDefinition;

  public FeedbackLoopHeaderTestCase(String name, PolicyDefinition policyDefinition) {
    this.policyDefinition = policyDefinition;
  }

  @Parameterized.Parameters
  public static Collection<Object[]> data() {
    return Arrays.asList(new Object[][] {
        {"Error continue", new PolicyDefinition(POLICY_ID, ERROR_BEFORE_FLOW_TEMPLATE_KEY, EMPTY_API_KEY, null, 1,
                                                new PolicyConfiguration(of("statusCode", STATUS_CODE)))},
        {"Error propagate", new PolicyDefinition(POLICY_ID, ERROR_PROPAGATE_TEMPLATE_KEY, EMPTY_API_KEY, null, 1,
                                                 new PolicyConfiguration(of("statusCode", STATUS_CODE)))}
    });
  }

  @BeforeClass
  public static void beforeClass() {
    fakeModel().createApi(EMPTY_APP2_API_KEY.id());
    installation.getServer().deployApplications(application2);
  }

  @Before
  public void setUp() {
    installation.getServer().assertApiTracked(EMPTY_APP2_API_KEY);
    installation.getServer().getPolicyDeploymentService()
        .newPolicy(policyDefinition);
    emptyRequest.withHeader("User-Agent", USER_AGENT);
    emptyRequestApp2.withHeader("User-Agent", USER_AGENT);
    fakeModel().clearEvents();
  }

  @After
  public void tearDown() {
    emptyRequest.cleanHeaders();
    emptyRequestApp2.cleanHeaders();
    installation.removePoliciesAndContext();
    fakeModel().clearEvents();
  }

  @Test
  public void noViaHeader() {
    HttpResponse response = emptyRequest.get();

    assertEquals(STATUS_CODE, response.statusCode());
    assertNull(response.header(FEEDBACK_LOOP_HEADER));
    checkAnalyticsViolation();
  }

  @Test
  public void isEdgeRequest() {
    HttpResponse response = emptyRequest.withHeader("via", VIA_EDGE_HEADER).get();

    assertEquals(STATUS_CODE, response.statusCode());
    assertEquals(CATEGORY, response.header(FEEDBACK_LOOP_HEADER));
    checkAnalyticsViolation();
  }

  @Test
  public void isEdgeRequestWithCorrelationId() {
    HttpResponse response = emptyRequest
        .withHeader("X-Correlation-Id", "55c332b5-0af7-4607-89c8-307105919545")
        .withHeader(VIA_HEADER_NAME, VIA_EDGE_HEADER)
        .get();

    assertEquals(STATUS_CODE, response.statusCode());
    assertEquals(CATEGORY, response.header(FEEDBACK_LOOP_HEADER));
    checkAnalyticsViolation();
  }

  @Test
  public void isAnyVersionEdgeRequest() {
    HttpResponse response = emptyRequest.withHeader(VIA_HEADER_NAME, VIA_EDGE_HEADER_ANY).get();

    assertEquals(STATUS_CODE, response.statusCode());
    assertEquals(CATEGORY, response.header(FEEDBACK_LOOP_HEADER));
    checkAnalyticsViolation();
  }

  @Test
  public void otherViaHeader() {
    HttpResponse response = emptyRequest.withHeader(VIA_HEADER_NAME, "HTTP/1.1 GWA").get();

    assertEquals(STATUS_CODE, response.statusCode());
    assertNull(response.header(FEEDBACK_LOOP_HEADER));
    checkAnalyticsViolation();
  }

  @Test
  public void multiViaHeader() {
    HttpResponse response = emptyRequest
        .withHeader(VIA_HEADER_NAME, "HTTP/1.1 GWA")
        .withHeader(VIA_HEADER_NAME, "HTTP/1.1 GWA2")
        .withHeader(VIA_HEADER_NAME, VIA_EDGE_HEADER)
        .withHeader(VIA_HEADER_NAME, "HTTP/1.1 GWA3")
        .get();

    assertEquals(STATUS_CODE, response.statusCode());
    assertEquals(CATEGORY, response.header(FEEDBACK_LOOP_HEADER));
    checkAnalyticsViolation();
  }

  @Test
  public void multiViaHeaderCommaSeparated() {
    HttpResponse response = emptyRequest
        .withHeader(VIA_HEADER_NAME, VIA_EDGE_HEADER + "," + VIA_EDGE_HEADER)
        .get();

    assertEquals(STATUS_CODE, response.statusCode());
    assertEquals(CATEGORY, response.header(FEEDBACK_LOOP_HEADER));
    checkAnalyticsViolation();
  }

  @Test
  public void policyViolationsInDifferentApps() {
    final int policy2StatusCode = 502;
    PolicyDefinition definitionApp2 =
        new PolicyDefinition(POLICY_ID_2, ERROR_BEFORE_FLOW_TEMPLATE_KEY_VIOLATION_CATEGORY2, EMPTY_APP2_API_KEY, null, 1,
                             new PolicyConfiguration(of("statusCode", policy2StatusCode)));
    installation.getServer().getPolicyDeploymentService()
        .newPolicy(definitionApp2);

    emptyRequest.withHeader(VIA_HEADER_NAME, VIA_EDGE_HEADER);
    emptyRequestApp2.withHeader(VIA_HEADER_NAME, VIA_EDGE_HEADER_ANY);

    HttpResponse response = emptyRequest.get();
    assertThat(response.statusCode(), is(STATUS_CODE));
    assertThat(response.header(FEEDBACK_LOOP_HEADER), is(CATEGORY));
    checkAnalyticsViolation();

    fakeModel().clearEvents();

    response = emptyRequestApp2.get();
    assertThat(response.statusCode(), is(policy2StatusCode));
    assertThat(response.header(FEEDBACK_LOOP_HEADER), is(CATEGORY_2));
    checkAnalyticsViolation(EMPTY_APP2_API_KEY, policy2StatusCode, EXPECTED_VIOLATION2);
  }

  private void checkAnalyticsViolation() {
    checkAnalyticsViolation(EMPTY_API_KEY, STATUS_CODE, EXPECTED_VIOLATION);
  }

  private void checkAnalyticsViolation(ApiKey apiKey, int statusCode, FakePolicyViolation violation) {
    probe(() -> {
      assertThat("Event was not present", fakeModel().getEvents(), hasSize(1));

      asserter()
          .methodIs(GET)
          .pathIs("/api/empty")
          .statusCodeIs(statusCode)
          .violationIs(violation)
          .apiVersionId(apiKey.id())
          .execute(fakeModel().getFirstEvent());
    });

  }

}
