/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.analytics.asserter;

import static com.mulesoft.anypoint.analytics.AbstractAnalyticsTestCase.USER_AGENT;
import static com.mulesoft.anypoint.tests.infrastructure.model.FakeApiModel.fakeModel;
import static com.mulesoft.anypoint.tests.infrastructure.model.FakeRequestDisposition.BLOCKED;
import static com.mulesoft.anypoint.tests.infrastructure.model.FakeRequestDisposition.PROCESSED;
import static java.util.Optional.ofNullable;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;
import static org.junit.Assert.assertThat;

import com.mulesoft.anypoint.tests.infrastructure.model.FakeApi;
import com.mulesoft.anypoint.tests.infrastructure.model.FakeHttpEvent;
import com.mulesoft.anypoint.tests.infrastructure.model.FakePolicyViolation;

import java.net.InetAddress;
import java.net.UnknownHostException;

public class AnalyticsEventAsserter {

  private String method;
  private String path;
  private int statusCode;
  private FakePolicyViolation policyViolation;

  private String apiName;
  private String apiVersion;
  private String instanceName;
  private String applicationName;
  private String applicationId;

  private int requestBytes;
  private boolean requestBytesSet;
  private int responseBytes;
  private boolean responseBytesSet;
  private Long apiKey;

  private AnalyticsEventAsserter() {}

  public static AnalyticsEventAsserter asserter() {
    return new AnalyticsEventAsserter();
  }

  public AnalyticsEventAsserter methodIs(String method) {
    this.method = method;
    return this;
  }

  public AnalyticsEventAsserter pathIs(String path) {
    this.path = path;
    return this;
  }

  public AnalyticsEventAsserter statusCodeIs(int statusCode) {
    this.statusCode = statusCode;
    return this;
  }

  public AnalyticsEventAsserter violationIs(FakePolicyViolation policyViolation) {
    this.policyViolation = policyViolation;
    return this;
  }

  public AnalyticsEventAsserter noViolation() {
    this.policyViolation = null;
    return this;
  }

  public AnalyticsEventAsserter apiNameIs(String apiName) {
    this.apiName = apiName;
    return this;
  }

  public AnalyticsEventAsserter apiVersionIs(String apiVersion) {
    this.apiVersion = apiVersion;
    return this;
  }

  public AnalyticsEventAsserter instanceNameIs(String instanceName) {
    this.instanceName = instanceName;
    return this;
  }

  public AnalyticsEventAsserter applicationNameIs(String applicationName) {
    this.applicationName = applicationName;
    return this;
  }

  public AnalyticsEventAsserter applicationIdIs(String applicationId) {
    this.applicationId = applicationId;
    return this;
  }

  public AnalyticsEventAsserter requestBytesAre(int requestBytes) {
    this.requestBytes = requestBytes;
    this.requestBytesSet = true;
    return this;
  }

  public AnalyticsEventAsserter responseBytesAre(int responseBytes) {
    this.responseBytes = responseBytes;
    this.responseBytesSet = true;
    return this;
  }

  public AnalyticsEventAsserter apiVersionId(Long apiKey) {
    this.apiKey = apiKey;
    return this;
  }

  public void execute(FakeHttpEvent event) {
    ofNullable(apiKey).ifPresent(
                                 id -> assertThat("Event was not produced by the expected api.", id,
                                                  is(event.getApiVersionId())));

    FakeApi fakeApi = fakeModel().getApi(event.getApiVersionId());

    assertThat("API was null", fakeApi, not(nullValue()));
    assertThat(event.getApiVersionId(), is(fakeApi.getId()));
    // Platform v3 API replaced old apiId with a new identifier, field legacyApiIdentifier
    // contains previous version identifier which is expected by analytics event.
    assertThat(event.getApiId(), is(fakeApi.getLegacyApiIdentifier()));
    assertThat(event.getOrgId(), is(fakeApi.getOrganizationId()));
    assertThat(event.getVerb(), is(method));
    assertThat(event.getPath(), is(path));
    assertThat(event.getUserAgent(), is(USER_AGENT));
    assertThat(event.getStatusCode(), is(statusCode));
    assertThat("Event ID was null", event.getEventId(), not(nullValue()));
    assertThat("Request bytes Ts was null", event.getHostId(), is(hostname()));
    assertThat("Received Ts was null", event.getReceivedTs(), not(nullValue()));
    assertThat("Replied Ts was null", event.getRepliedTs(), not(nullValue()));
    assertThat("Client IP was null", event.getClientIp(), not(nullValue()));
    assertThat("Request bytes was null", event.getRequestBytes(), not(nullValue()));

    if (requestBytesSet) {
      assertThat("Request bytes differs from what expected", event.getRequestBytes(), is(requestBytes));
    }

    if (responseBytesSet) {
      assertThat("Response bytes differs from what expected", event.getResponseBytes(), is(responseBytes));
    }

    assertThat("Client id should be set by analytics", event.getClientId(), is(applicationId));

    // On-prem only properties
    assertThat(event.getApiName(), is(apiName));
    assertThat(event.getApiVersion(), is(apiVersion));
    assertThat(event.getInstanceName(), is(instanceName));
    assertThat(event.getApplicationName(), is(applicationName));

    if (policyViolation != null) {
      assertThat("Policy Violation was null", event.getPolicyViolation(), not(nullValue()));
      assertThat(event.getPolicyViolation().getPolicyId(), is(policyViolation.getPolicyId()));
      assertThat(event.getPolicyViolation().getOutcome(), is(policyViolation.getOutcome()));
      assertThat(event.getPolicyViolation().getPolicyName(), is(policyViolation.getPolicyName()));
      assertThat(event.getRequestDisposition(), is(BLOCKED));
    } else {
      assertThat("Policy Violation was not null", event.getPolicyViolation(), nullValue());
      assertThat(event.getRequestDisposition(), is(PROCESSED));
    }
  }

  private String hostname() {
    try {
      return InetAddress.getLocalHost().getHostName();
    } catch (UnknownHostException e) {
      throw new RuntimeException(e);
    }
  }
}
