/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.tests.infrastructure;

import static com.google.common.collect.Lists.newArrayList;
import static com.mulesoft.anypoint.tita.environment.maven.MavenInstallation.getDirectDependencies;
import static com.mulesoft.anypoint.tita.environment.maven.MavenInstallation.resolveArtifact;
import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toList;
import static org.mule.runtime.core.api.config.MuleManifest.getProductVersion;
import static org.mule.runtime.core.api.util.FileUtils.unzip;

import org.mule.runtime.container.api.MuleCoreExtension;

import com.mulesoft.anypoint.tita.environment.api.artifact.PolicyJar;
import com.mulesoft.mule.runtime.gw.deployment.ApiDeploymentCoreExtension;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.module.cluster.internal.HazelcastClusterCoreExtension;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import org.eclipse.aether.artifact.Artifact;
import org.eclipse.aether.graph.Dependency;

public class FakeGatewayServerBuilder {

  private static final String GATEWAY_INFRASTRUCTURE = "api-gateway-tests-infrastructure";

  private static final String EE_DISTRIBUTION = "com.mulesoft.mule.distributions";
  private static final String CE_DISTRIBUTION = "org.mule.distributions";

  private static final String SERVICES_ALL = "mule-services-all";
  private static final String GATEWAY_BOM = "api-gateway-bom-impl";

  private static final String GATEWAY_SERVICES = "com.mulesoft.anypoint";
  private static final String ORG_MULE_SERVICES = "org.mule.services";
  private static final String COM_MULESOFT_MULE_SERVICES = "com.mulesoft.mule.services";
  private static final String COM_MULESOFT_SERVICES = "com.mulesoft.services";

  private static final String MULE_SERVICE_HTTP = "mule-service-http-ee";
  private static final String MULE_SERVICE_SCHEDULER = "mule-service-scheduler";
  private static final String MULE_SERVICE_WEAVE = "mule-service-weave-ee";
  private static final String GATEWAY_SERVICE_CONTRACTS = "api-gateway-contract-service";
  private static final String GATEWAY_SERVICE_EVENTS = "api-gateway-events-collector-service";

  private static final String POM = "pom";

  private String muleHome;
  private List<Class<? extends MuleCoreExtension>> coreExtensions = initialCoreExtensions();
  private List<PolicyJar> policyTemplates = new ArrayList<>();
  private List<PolicyDefinition> policyDefinitions = new ArrayList<>();

  private String clusterId = UUID.randomUUID().toString();

  public FakeGatewayServerBuilder withMuleHome(String muleHome) {
    this.muleHome = muleHome;
    return this;
  }

  public FakeGatewayServerBuilder clusterId(String clusterId) {
    this.clusterId = clusterId;
    return this;
  }

  public FakeGatewayServerBuilder addCoreExtension(Class<? extends MuleCoreExtension> coreExtension) {
    this.coreExtensions.add(coreExtension);
    return this;
  }

  public FakeGatewayServerBuilder removeCoreExtension(Class<? extends MuleCoreExtension> coreExtensionClass) {
    this.coreExtensions.remove(coreExtensionClass);
    return this;
  }

  public FakeGatewayServerBuilder withPolicyTemplates(PolicyJar... policyTemplates) {
    this.policyTemplates = asList(policyTemplates);
    return this;
  }

  public void withPolicyDefinitions(PolicyDefinition... policyDefinitions) {
    this.policyDefinitions = asList(policyDefinitions);
  }

  public FakeGatewayServer build() {
    List<MuleCoreExtension> coreExtensionInstances = coreExtensions.stream().map(this::instanceCoreExtension).collect(toList());

    FakeGatewayServer server = new FakeGatewayServer(muleHome, coreExtensionInstances, clusterId);

    policyTemplates.forEach(server::installPolicyTemplate);
    policyDefinitions.forEach(server::storePolicyDefinition);

    addServices(server);

    return server;
  }

  private void addServices(FakeGatewayServer server) {
    List<Dependency> infra = getDirectDependencies(GATEWAY_SERVICES, GATEWAY_INFRASTRUCTURE, getProductVersion(), null, null);

    try {
      unzipService(server, artifact(infra, ORG_MULE_SERVICES, MULE_SERVICE_SCHEDULER));
      unzipService(server, artifact(infra, COM_MULESOFT_SERVICES, MULE_SERVICE_WEAVE));
      unzipService(server, artifact(infra, COM_MULESOFT_MULE_SERVICES, MULE_SERVICE_HTTP));
      unzipService(server, artifact(infra, GATEWAY_SERVICES, GATEWAY_SERVICE_CONTRACTS));
      unzipService(server, artifact(infra, GATEWAY_SERVICES, GATEWAY_SERVICE_EVENTS));
    } catch (IOException e) {
      throw new RuntimeException("Error adding services to Fake Gateway Server", e);
    }
  }

  private Artifact artifact(List<Dependency> dependencies, String groupId, String artifactId) {
    return dependencies.stream()
        .filter(dependency -> dependencyMatches(dependency, groupId, artifactId))
        .findFirst()
        .orElseThrow(() -> new RuntimeException("Dependency " + artifactId + " not found in mule-services-all"))
        .getArtifact();
  }

  private void unzipService(FakeGatewayServer server, Artifact artifact) throws IOException {
    File serviceFolder = new File(server.getServicesDir(), serviceFileName(artifact));
    unzip(resolveArtifact(artifact.getGroupId(), artifact.getArtifactId(), artifact.getVersion(), artifact.getClassifier(),
                          artifact.getExtension()),
          serviceFolder);
  }

  private String serviceFileName(Artifact artifact) {
    return artifact.getArtifactId() + "-" + artifact.getVersion() + "-" + artifact.getClassifier();
  }

  private boolean dependencyMatches(Dependency dependency, String groupId, String artifactId) {
    return dependency.getArtifact().getArtifactId().equals(artifactId) && dependency.getArtifact().getGroupId().equals(groupId);
  }

  private MuleCoreExtension instanceCoreExtension(Class<? extends MuleCoreExtension> aClass) {
    try {
      return aClass.newInstance();
    } catch (InstantiationException | IllegalAccessException e) {
      throw new RuntimeException(e);
    }
  }

  private List<Class<? extends MuleCoreExtension>> initialCoreExtensions() {
    return newArrayList(HazelcastClusterCoreExtension.class, ApiDeploymentCoreExtension.class);
  }
}
