/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.tests.infrastructure.installation;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.io.Files.createTempDir;
import static com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallationConfiguration.GatewayMode.OFFLINE;
import static com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallationConfiguration.GatewayMode.ONLINE;
import static com.mulesoft.mule.runtime.gw.api.config.GateKeeperConfiguration.Mode.DISABLED;
import static com.mulesoft.mule.runtime.gw.api.config.GateKeeperConfiguration.Mode.FLEXIBLE;
import static com.mulesoft.mule.runtime.gw.api.config.GateKeeperConfiguration.Mode.STRICT;
import static com.mulesoft.mule.runtime.gw.api.config.OnApiDeletedConfiguration.Mode.KEEP_POLICIES;
import static java.util.Arrays.asList;
import static java.util.stream.IntStream.range;

import com.mulesoft.anypoint.tests.infrastructure.FakeGatewayServerBuilder;
import com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallationConfiguration.GatewayMode;
import com.mulesoft.anypoint.tests.infrastructure.rules.ClusterDynamicPort;
import com.mulesoft.anypoint.tita.environment.api.artifact.Artifact;
import com.mulesoft.anypoint.tita.environment.api.artifact.PolicyJar;
import com.mulesoft.mule.runtime.gw.analytics.AnalyticsCoreExtension;
import com.mulesoft.mule.runtime.gw.api.config.GateKeeperConfiguration.Mode;
import com.mulesoft.mule.runtime.gw.api.config.OnApiDeletedConfiguration;
import com.mulesoft.mule.runtime.gw.deployment.ApiDeploymentCoreExtension;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;

import java.io.File;
import java.util.List;
import java.util.UUID;

public abstract class AbstractInstallationBuilder<T extends AbstractInstallationBuilder> {

  List<FakeGatewayServerBuilder> serverBuilders = newArrayList();
  List<Artifact> applications = newArrayList();
  List<Artifact> domains = newArrayList();
  ClusterDynamicPort dynamicPort;
  GatewayMode gatewayMode = ONLINE;
  boolean encrypted;
  String encryptionKey;
  boolean clientModeEnabled;
  Mode gateKeeperMode = STRICT;
  JdbcStoreConfiguration jdbcStoreConfiguration;
  OnApiDeletedConfiguration.Mode onApiDeleted = KEEP_POLICIES;

  public AbstractInstallationBuilder(int size, int numberOfGroups) {
    File baseDirectory = createTempDir();
    range(0, numberOfGroups)
        .forEach(numberOfGroup -> {
          String clusterId = UUID.randomUUID().toString();
          range(0, size).forEach(nodeId -> {
            FakeGatewayServerBuilder serverBuilder =
                new FakeGatewayServerBuilder()
                    .withMuleHome(new File(baseDirectory, getMuleFolderName(clusterId, nodeId)).getAbsolutePath())
                    .clusterId(clusterId);
            serverBuilders.add(serverBuilder);
          });
        });
  }

  private String getMuleFolderName(String clusterId, int nodeId) {
    return clusterId + "-" + String.valueOf(nodeId);
  }

  /**
   * Sets the initial apps that will be deployed when server starts.
   *
   * @param applications the apps
   * @return this
   */
  public T withApplications(Artifact... applications) {
    checkNotNull(applications);
    this.applications = asList(applications);
    return (T) this;
  }

  public AbstractInstallationBuilder<T> withDomains(Artifact... domains) {
    checkNotNull(domains);
    this.domains = asList(domains);
    return (T) this;
  }

  /**
   * Sets the clientMode.
   *
   * @param clientModeEnabled whether the installation ahs to be as client.
   * @return this
   */
  public T withClientModeEnabled(boolean clientModeEnabled) {
    this.clientModeEnabled = clientModeEnabled;
    return (T) this;
  }

  /**
   * Sets the initial templates that will be present in policy-templates folder when server starts.
   *
   * @param policyTemplates the templates
   * @return this
   */
  public T withPolicyTemplates(PolicyJar... policyTemplates) {
    checkNotNull(policyTemplates);
    serverBuilders.forEach(serverBuilder -> serverBuilder.withPolicyTemplates(policyTemplates));
    return (T) this;
  }

  /**
   * Will start the server with analytics functionality
   *
   * @return this
   */
  public T analyticsEnabled() {
    serverBuilders.forEach(serverBuilder -> serverBuilder.addCoreExtension(AnalyticsCoreExtension.class));
    return (T) this;
  }

  /**
   * Will start the server without client id and secret being set. Also known as offline mode.
   *
   * @return this
   */
  public T offline() {
    this.gatewayMode = OFFLINE;
    return (T) this;
  }

  /**
   * Will start the server without starting Gateway plugin.
   *
   * @return this
   */
  public T disabled() {
    this.gatewayMode = GatewayMode.DISABLED;
    return (T) this;
  }

  /**
   * Will start the server with GateKeeper disabled
   *
   * @return this
   */
  public T gateKeeperDisabled() {
    this.gateKeeperMode = DISABLED;
    return (T) this;
  }

  /**
   * Will start the server with Client ID and Secret properties encrypted
   *
   * @return this
   */
  public T withEncryptedProperties() {
    this.encrypted = true;
    return (T) this;
  }

  /**
   * Will start the server with the provided encryption key
   *
   * @return this
   */
  public T withEncryptionKey(String key) {
    this.encryptionKey = key;
    return (T) this;
  }

  /**
   * Will start the server with GateKeeper disabled
   *
   * @return this
   */
  public T flexibleGateKeeper() {
    this.gateKeeperMode = FLEXIBLE;
    return (T) this;
  }

  public T withJdbcStore(JdbcStoreConfiguration jdbcStoreConfiguration) {
    this.jdbcStoreConfiguration = jdbcStoreConfiguration;
    return (T) this;
  }

  /**
   * Set the port provider for cluster. This is for avoid port clashing between different nodes.
   *
   * @param dynamicPort the port provider
   * @return this
   */
  public T withDynamicPort(ClusterDynamicPort dynamicPort) {
    this.dynamicPort = dynamicPort;
    return (T) this;
  }

  /**
   * Will start the server without policy deployment functionality. This also implies that the server will be in offline mode
   *
   * @return this
   */
  public T disablePolicyDeployment() {
    serverBuilders.forEach(serverBuilder -> serverBuilder.removeCoreExtension(ApiDeploymentCoreExtension.class));
    return (T) this;
  }

  /**
   * Sets the initial policies that will be present in policies folder when server starts.
   *
   * @param policyDefinitions the policy definitions
   * @return this
   */
  public T withPolicyDefinitions(PolicyDefinition... policyDefinitions) {
    serverBuilders.forEach(serverBuilder -> serverBuilder.withPolicyDefinitions(policyDefinitions));
    return (T) this;
  }

  /**
   *
   * @param mode
   * @return
   */
  public T onApiDeleted(OnApiDeletedConfiguration.Mode mode) {
    this.onApiDeleted = mode;
    return (T) this;
  }

  public abstract Installation build();
}
