/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.tests.infrastructure;

import static java.util.Arrays.asList;
import static org.apache.commons.io.FileUtils.copyFile;
import static org.apache.commons.io.FileUtils.copyFileToDirectory;
import static org.apache.commons.io.FileUtils.deleteQuietly;
import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertThat;

import org.mule.runtime.container.api.MuleCoreExtension;
import org.mule.runtime.deployment.model.api.application.Application;
import org.mule.runtime.module.deployment.api.DeploymentService;
import org.mule.runtime.module.service.api.manager.ServiceManager;
import org.mule.tck.probe.PollingProber;
import org.mule.test.infrastructure.deployment.FakeMuleServer;

import com.mulesoft.anypoint.tests.DescriptiveProbe;
import com.mulesoft.anypoint.tests.infrastructure.exception.AppDeploymentException;
import com.mulesoft.anypoint.tests.infrastructure.exception.AppUndeploymentException;
import com.mulesoft.anypoint.tita.environment.api.artifact.Artifact;
import com.mulesoft.anypoint.tita.environment.api.artifact.PolicyJar;
import com.mulesoft.mule.runtime.gw.api.agent.HealthCheck;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.api.service.ContractService;
import com.mulesoft.anypoint.backoff.scheduler.BackoffScheduler;
import com.mulesoft.mule.runtime.gw.client.provider.ApiPlatformClientProvider;
import com.mulesoft.mule.runtime.gw.deployment.ApiDeploymentCoreExtension;
import com.mulesoft.mule.runtime.gw.deployment.ApiService;
import com.mulesoft.mule.runtime.gw.deployment.contracts.ContractSnapshots;
import com.mulesoft.mule.runtime.gw.deployment.platform.interaction.PlatformInteractionLifecycle;
import com.mulesoft.mule.runtime.gw.deployment.platform.interaction.PlatformInteractionServicesLifecycle;
import com.mulesoft.mule.runtime.gw.deployment.platform.interaction.apis.ApisPollerPlatformInteractionLifecycle;
import com.mulesoft.mule.runtime.gw.deployment.platform.interaction.clients.ClientsPlatformInteractionLifecycleAdapter;
import com.mulesoft.mule.runtime.gw.deployment.platform.interaction.keepalive.KeepAlivePlatformInteractionLifecycleAdapter;
import com.mulesoft.mule.runtime.gw.deployment.tracking.ApiTrackingService;
import com.mulesoft.mule.runtime.gw.hdp.listener.HdpDeploymentListener;
import com.mulesoft.mule.runtime.gw.metrics.GatewayMetricsAdapter;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.model.PolicySet;
import com.mulesoft.mule.runtime.gw.policies.Policy;
import com.mulesoft.mule.runtime.gw.policies.factory.PolicyFactory;
import com.mulesoft.mule.runtime.gw.policies.service.PolicyDeploymentService;
import com.mulesoft.mule.runtime.gw.policies.service.PolicySetDeploymentService;
import com.mulesoft.mule.runtime.gw.policies.store.DefaultPolicyStore;
import com.mulesoft.mule.runtime.gw.policies.store.EncryptedPropertiesSerializer;
import com.mulesoft.mule.runtime.gw.policies.store.PolicyStore;
import com.mulesoft.mule.runtime.gw.reflection.Inspector;

import com.google.common.collect.ImmutableList;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.List;
import java.util.Optional;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.reflect.FieldUtils;
import org.hamcrest.core.Is;

public class FakeGatewayServer extends FakeMuleServer {

  private final PolicyStore policyStore;
  private final File policiesDir;
  private final File hdpRegistryDir;
  private final String clusterId;

  public FakeGatewayServer(String muleHomePath, List<MuleCoreExtension> initialCoreExtensions, String clusterId) {
    super(muleHomePath, initialCoreExtensions);
    this.policiesDir = createFolder("policies");
    this.hdpRegistryDir = createFolder("policies/hdp-api-registry");
    this.policyStore = new DefaultPolicyStore(new EncryptedPropertiesSerializer());
    this.clusterId = clusterId;
  }

  public File getPoliciesDir() {
    return this.policiesDir;
  }

  public <T extends MuleCoreExtension> T getCoreExtension(Class<T> coreExtensionClass) {
    List<MuleCoreExtension> coreExtensions = new Inspector(this).read("coreExtensions");

    return (T) coreExtensions.stream()
        .filter(c -> c.getClass().isAssignableFrom(coreExtensionClass))
        .findFirst()
        .orElseThrow(() -> new IllegalArgumentException("Core Extension not present in Server"));
  }

  // TODO make this private
  public DeploymentService getDeploymentService() {
    try {
      return (DeploymentService) FieldUtils.readField(this, "deploymentService", true);
    } catch (IllegalAccessException e) {
      throw new RuntimeException(e);
    }
  }

  /**
   * Installs the given applications and verifies that they were correctly deployed
   */
  public FakeGatewayServer deployApplications(Artifact... applications) {
    ImmutableList.copyOf(applications).forEach(app -> {
      try {
        installApplication(app);
        assertDeploymentSuccess(app.getName());
      } catch (AssertionError e) {
        try {
          assertDeploymentSuccess(app.getName());
        } catch (AssertionError e2) {
          throw new AppDeploymentException("App " + app.getName() + " could not be correctly deployed");
        }
      }
    });
    return this;
  }

  /**
   * Installs the given applications and verifies that the deployment failed
   */
  public FakeGatewayServer deployApplicationsExpectingFailure(Artifact... applications) {
    ImmutableList.copyOf(applications).forEach(app -> {
      try {
        installApplication(app);
        assertDeploymentFailure(app.getName());
      } catch (AssertionError e) {
        throw new AppDeploymentException("App " + app.getName() + " was expected to cause a deployment failure but it did not");
      }
    });
    return this;
  }

  /**
   * Copies the given applications to the applications folder
   */
  public void installApplications(List<Artifact> applications) {
    applications.forEach(this::installApplication);
  }

  public void installDomains(List<Artifact> domains) {
    domains.forEach(this::installDomain);
  }

  private void installDomain(Artifact domain) {
    try {
      copyFileToDirectory(domain.getJarFile(), new File(getMuleHome(), "domains"));
    } catch (IOException e) {
      throw new AppDeploymentException("Error copying domain jar to domains dir", e);
    }
  }

  /**
   * Undeploy application through anchor file deletion instead of directly using deployment service, since the latter is producing
   * flaky behaviour because it uses non-blocking locks and has no retry mechanism in it.
   */
  public FakeGatewayServer undeployApplication(String appName) {
    resetDeploymentListener();
    Application application = getDeploymentService().findApplication(appName);
    File anchor = new File(application.getLocation().getAbsolutePath() + "-anchor.txt");

    if (!anchor.exists()) {
      undeployFailedApplication(application);
    } else if (!anchor.delete()) {
      throw new AppUndeploymentException("Error trying to delete anchor file of App " + appName);
    }

    try {
      assertUndeploymentSuccess(appName);
    } catch (AssertionError e) {
      throw new AppUndeploymentException("App " + appName + " was not correctly undeployed");
    }

    return this;
  }

  public FakeGatewayServer deployPolicy(Policy policy) {
    getPolicyDeploymentService().newPolicy(policy.getPolicyDefinition());
    return this;
  }

  public FakeGatewayServer deployPolicy(PolicyDefinition policyDefinition) {
    getPolicyDeploymentService().newPolicy(policyDefinition);
    return this;
  }

  public FakeGatewayServer deployPoliciesForApi(ApiKey apiKey, PolicySet policySet) {
    getPolicySetDeploymentService().policiesForApi(apiKey, policySet);
    return this;
  }

  public FakeGatewayServer undeployPolicy(String name) {
    getPolicyDeploymentService().removePolicy(name);
    return this;
  }

  public FakeGatewayServer removeAllPoliciesAndContext() {
    allPolicies().forEach(policyDefinition -> getPolicyDeploymentService().removePolicy(policyDefinition.getName()));

    cleanUpDotMule();
    return this;
  }

  public void assertApiTracked(ApiKey apiKey) {
    new PollingProber(20000, 100).check(new DescriptiveProbe(() -> {
      assertThat(getApiService().get(apiKey).isPresent(), Is.is(true));
      assertThat(getApiService().get(apiKey).get().getTrackingInfo().isTracked(), is(true));
    }, "API " + apiKey + " was not tracked"));
  }

  public FakeGatewayServer updateHdpRegistry(String source, String target) {
    try {
      URL sourceUri = this.getClass().getClassLoader().getResource(source);
      if (sourceUri == null) {
        throw new RuntimeException("Resource not found: " + source);
      }
      FileUtils.copyFile(new File(sourceUri.toURI()), new File(hdpRegistryDir, target));
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
    return this;
  }

  public ContractService getContractService() {
    ServiceManager serviceManager = new Inspector(this).read("serviceManager");
    return (ContractService) serviceManager.getServices()
        .stream()
        .filter(service -> ContractService.class.isAssignableFrom(service.getClass()))
        .findFirst()
        .orElseThrow(() -> new RuntimeException("Fail to get ContractService"));
  }

  public List<PolicyDefinition> allPolicies() {
    return policyStore.load();
  }

  public void installPolicyTemplate(PolicyJar policyTemplate) {

    File policyTemplatesFolder = new File(getPoliciesDir(), "policy-templates");
    if (!policyTemplatesFolder.exists() && !policyTemplatesFolder.mkdir()) {
      throw new RuntimeException("Could not create policy templates folder");
    }

    try {
      copyFileToDirectory(policyTemplate.getJarFile(), policyTemplatesFolder);
    } catch (IOException e) {
      throw new RuntimeException("Could not copy template jar file " + policyTemplate.getName(), e);
    }

    policyTemplate.getYamlFile().ifPresent(file -> {
      try {
        copyFile(file, new File(policyTemplatesFolder, policyTemplate.key().getName() + ".yaml"));
      } catch (IOException e) {
        throw new RuntimeException("Could not copy template yaml file " + policyTemplate.getName(), e);
      }
    });
  }

  public void storePolicyDefinition(PolicyDefinition policyDefinition) {
    this.policyStore.store(policyDefinition);
  }

  public PolicyFactory getPolicyFactory() {
    return new Inspector(getPolicySetDeploymentService()).read("policyFactory");
  }

  public PolicyDeploymentService getPolicyDeploymentService() {
    return new Inspector(getPolicySetDeploymentService()).read("policyDeploymentService");
  }

  public PolicySetDeploymentService getPolicySetDeploymentService() {
    return new Inspector(getCoreExtension(ApiDeploymentCoreExtension.class)).read("policySetDeploymentService");
  }

  public ApiTrackingService getApiTrackingService() {
    return new Inspector(getCoreExtension(ApiDeploymentCoreExtension.class))
        .read("apiTrackingService");
  }

  public ApiService getApiService() {
    return getCoreExtension(ApiDeploymentCoreExtension.class).getApiService();
  }

  public HdpDeploymentListener getHdpDeploymentListener() {
    return new Inspector(getCoreExtension(ApiDeploymentCoreExtension.class))
        .read("hdpDeploymentListener");
  }

  public ContractSnapshots getContractSnapshots() {
    return new Inspector(getCoreExtension(ApiDeploymentCoreExtension.class))
        .read("contractSnapshots");
  }

  public BackoffScheduler getClientsPollersManagerScheduler() {
    ApiDeploymentCoreExtension coreExtension = getCoreExtension(ApiDeploymentCoreExtension.class);
    return new Inspector(coreExtension)
        .read("platformInteractionServicesLifecycle.clientsPlatformInteractionLifecycle.lifecycle.poller.scheduler");
  }

  public BackoffScheduler getApisPollersManagerSchduler() {
    ApiDeploymentCoreExtension coreExtension = getCoreExtension(ApiDeploymentCoreExtension.class);
    return new Inspector(coreExtension)
        .read("platformInteractionServicesLifecycle.apisPlatformInteractionLifecycle.pollerLifecycle.pollerLifecycle.poller.scheduler");
  }

  public BackoffScheduler getKeepAlivePollersManagerScheduler() {
    ApiDeploymentCoreExtension coreExtension = getCoreExtension(ApiDeploymentCoreExtension.class);
    return new Inspector(coreExtension)
        .read("platformInteractionServicesLifecycle.keepAlivePlatformInteractionLifecycle.lifecycle.pollerLifecycle.poller.scheduler");
  }

  public boolean apiPollerStarted() {
    ApiDeploymentCoreExtension coreExtension = getCoreExtension(ApiDeploymentCoreExtension.class);
    PlatformInteractionLifecycle apisPlatformInteractionLifecycle =
        new Inspector(coreExtension).read("platformInteractionServicesLifecycle.apisPlatformInteractionLifecycle");
    return (apisPlatformInteractionLifecycle instanceof ApisPollerPlatformInteractionLifecycle)
        && (getApisPollersManagerSchduler() != null);
  }

  public boolean keepAlivePollerStarted() {
    ApiDeploymentCoreExtension coreExtension = getCoreExtension(ApiDeploymentCoreExtension.class);
    PlatformInteractionLifecycle keepAlivePlatformInteractionLifecycle =
        new Inspector(coreExtension).read("platformInteractionServicesLifecycle.keepAlivePlatformInteractionLifecycle");
    return (keepAlivePlatformInteractionLifecycle instanceof KeepAlivePlatformInteractionLifecycleAdapter)
        && (getKeepAlivePollersManagerScheduler() != null);
  }

  public boolean clientsPollerStarted() {
    ApiDeploymentCoreExtension coreExtension = getCoreExtension(ApiDeploymentCoreExtension.class);
    PlatformInteractionLifecycle clientsPlatformInteractionLifecycle =
        new Inspector(coreExtension).read("platformInteractionServicesLifecycle.clientsPlatformInteractionLifecycle");
    return (clientsPlatformInteractionLifecycle instanceof ClientsPlatformInteractionLifecycleAdapter)
        && (getClientsPollersManagerScheduler() != null);
  }


  public boolean platformPollersStarted() {
    return apiPollerStarted() || keepAlivePollerStarted() || clientsPollerStarted();
  }

  public PlatformInteractionServicesLifecycle getPlatformInteractionLifecycle() {
    ApiDeploymentCoreExtension coreExtension = getCoreExtension(ApiDeploymentCoreExtension.class);
    return new Inspector(coreExtension).read("platformInteractionServicesLifecycle");
  }

  public void forceGatewayStatusMetric() {
    ApiDeploymentCoreExtension coreExtension = getCoreExtension(ApiDeploymentCoreExtension.class);
    new Inspector(coreExtension)
        .<Optional<GatewayMetricsAdapter>>read("metricsCollector")
        .ifPresent(GatewayMetricsAdapter::generateStatusEvent);
  }

  public HealthCheck getHealthCheck() {
    ApiDeploymentCoreExtension coreExtension = getCoreExtension(ApiDeploymentCoreExtension.class);
    return coreExtension.healthCheck();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    FakeGatewayServer that = (FakeGatewayServer) o;

    return getMuleHome().equals(that.getMuleHome());
  }

  @Override
  public int hashCode() {
    return getMuleHome().hashCode();
  }

  private void cleanUpDotMule() {
    File dotMule = new File(getMuleHome(), ".mule");

    asList(dotMule.listFiles()).forEach(file -> {
      if (!isServicesFolder(file) && !isTemplatesFolder(file)) {
        deleteQuietly(file);
      }
    });
  }

  private void installApplication(Artifact app) {
    try {
      copyFileToDirectory(app.getJarFile(), getAppsDir());
    } catch (IOException e) {
      throw new AppDeploymentException("Error copying app jar to apps dir", e);
    }
  }

  private boolean isTemplatesFolder(File file) {
    return file.isDirectory() && file.getName().equals("policy-templates");
  }

  private boolean isServicesFolder(File file) {
    return file.isDirectory() && file.getName().equals("services");
  }

  private File createFolder(String folderName) {
    File folder = new File(this.getMuleHome(), folderName);
    if (!folder.exists() && !folder.mkdirs()) {
      throw new IllegalStateException(String.format("Unable to create folder '%s'", folderName));
    }

    return folder;
  }

  /**
   * Undeploys a failed application which does not contains an anchor file
   */
  private void undeployFailedApplication(Application application) {
    File appFolder = new File(application.getLocation().getAbsolutePath());
    if (!appFolder.exists()) {
      throw new AppUndeploymentException("Application folder nor anchor file are present for App "
          + application.getArtifactName());
    }

    try {
      FileUtils.deleteDirectory(appFolder);
    } catch (IOException e) {
      throw new AppUndeploymentException("Error trying to delete app folder (anchor file was not present) of App" +
          application.getArtifactName(), e);
    }
  }

  public String getClusterId() {
    return clusterId;
  }


}
