/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.tests.infrastructure.installation;

import static java.util.stream.Collectors.toList;
import static org.apache.commons.io.FileUtils.deleteQuietly;
import static org.junit.Assert.fail;
import static org.mule.runtime.core.internal.util.FunctionalUtils.safely;

import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.core.api.config.MuleProperties;
import org.mule.tck.probe.PollingProber;
import org.mule.tck.probe.Probe;

import com.mulesoft.anypoint.tests.infrastructure.FakeGatewayServer;
import com.mulesoft.anypoint.tests.infrastructure.FakeGatewayServerBuilder;
import com.mulesoft.anypoint.tests.infrastructure.rules.ClusterDynamicPort;
import com.mulesoft.anypoint.tita.environment.api.artifact.Artifact;
import com.mulesoft.mule.runtime.module.cluster.internal.ClusterSupportProperties;
import com.mulesoft.mule.runtime.module.cluster.internal.HazelcastClusterCoreExtension;
import com.mulesoft.mule.runtime.module.cluster.internal.HazelcastClusterManager;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.junit.rules.ExternalResource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * JUnit rule that set up a cluster of FakeGatewayServer and deploys the selected applications to every node.
 */
public class FakeGatewayClusterInstallation extends ExternalResource implements Installation<FakeGatewayClusterInstallation> {

  private static final int DEFAULT_NODES = 2;
  private static final Logger LOGGER = LoggerFactory.getLogger(FakeGatewayClusterInstallation.class);

  private List<FakeGatewayServer> servers = new ArrayList<>();
  private List<Artifact> applications = new ArrayList<>();
  private ClusterDynamicPort dynamicPort;
  private ClusterSystemProperties clusterSystemProperties;
  private FakeGatewayInstallationConfiguration configuration;
  private boolean clientMode;
  private int numberOfGroups = 1;
  private Builder builder;

  public FakeGatewayClusterInstallation(List<Artifact> applications, ClusterDynamicPort dynamicPort,
                                        FakeGatewayInstallationConfiguration configuration,
                                        boolean clientMode, int numberOfGroups, Builder builder) {
    updateState(servers, applications, dynamicPort, configuration, clientMode, numberOfGroups, builder);

  }

  public static Builder builder() {
    return new Builder(DEFAULT_NODES, 1);
  }

  public static Builder withSize(int size) {
    return new Builder(size, 1);
  }

  public static Builder builder(int numberOfGroups) {
    return new Builder(DEFAULT_NODES, numberOfGroups);
  }

  @Override
  protected void before() throws Throwable {
    this.servers = builder.buildServers();
    configuration.before();
    try {
      spinUpServers();
    } catch (Throwable t) {
      after();
      throw t;
    }
  }

  @Override
  protected void after() {
    configuration.after();
    clusterSystemProperties.after();
    dynamicPort.dispose();
    servers.forEach(server -> {
      stopQuietly(server);
      deleteQuietly(server.getMuleHome());
    });
  }

  @Override
  public FakeGatewayClusterInstallation restart() {
    safely(() -> {
      for (FakeGatewayServer server : servers) {
        stopQuietly(server);
      }
      dynamicPort.dispose();
      replicate(builder.build());
      this.servers = builder.buildServers();
      spinUpServers();
    }, error -> {
      error.printStackTrace();
      fail(restartMessage(error));
    });
    return this;
  }

  public FakeGatewayServer getNode(int i) {
    return servers.get(i);
  }

  private void startClusterNodes(int nodeIdSeed, List<FakeGatewayServer> servers) throws Exception {
    int i = nodeIdSeed;
    for (FakeGatewayServer server : servers) {
      System.setProperty(ClusterSupportProperties.CLUSTER_ID_PROPERTY_KEY, server.getClusterId());
      startNode(server, i++);
    }
  }

  private void startNode(FakeGatewayServer server, int nodeId) throws IOException, MuleException {
    // In order to be able to deploy apps, we first need to set the MULE_HOME property to the server that is doing the deploy
    System.setProperty(MuleProperties.MULE_HOME_DIRECTORY_PROPERTY, server.getMuleHome().getAbsolutePath());
    clusterSystemProperties.updateClusterNodeIdProperty(String.valueOf(nodeId));

    if (dynamicPort != null) {
      dynamicPort.createPortForServer(server);
    }

    server.installApplications(applications);
    server.start();
    applications.forEach(app -> server.assertDeploymentSuccess(app.getName()));
  }

  private void stopQuietly(FakeGatewayServer server) {
    try {
      server.stop();
    } catch (Throwable e) {
      LOGGER.error("Stop failed, reason: " + e.getMessage());
    }
  }

  public FakeGatewayClusterInstallation whileNodeIsStopped(int ithNode, Runnable closure) {
    try {
      stopQuietly(getNode(ithNode));
      closure.run();
    } catch (Exception e) {
      fail("Closure execution failed: " + temporalStopMessage(e));
    } finally {
      restart();
    }
    return this;
  }

  @Override
  public FakeGatewayClusterInstallation removePoliciesAndContext() {
    servers.forEach(FakeGatewayServer::removeAllPoliciesAndContext);
    return this;
  }

  private void spinUpServers() throws Exception {
    spinUpServers(1, servers);
  }

  private void spinUpServers(int nodeIdSeed, List<FakeGatewayServer> servers) throws Exception {
    clusterSystemProperties.before();
    startClusterNodes(nodeIdSeed, servers);
    assertClusterStarted();
  }

  private void replicate(FakeGatewayClusterInstallation newInstallation) {
    updateState(newInstallation.servers,
                newInstallation.applications,
                newInstallation.dynamicPort,
                newInstallation.configuration,
                newInstallation.clientMode,
                newInstallation.numberOfGroups,
                newInstallation.builder);
  }

  private void updateState(List<FakeGatewayServer> servers, List<Artifact> applications,
                           ClusterDynamicPort dynamicPort, FakeGatewayInstallationConfiguration configuration,
                           boolean clientMode, int numberOfGroups, Builder builder) {
    this.servers = servers;
    this.dynamicPort = dynamicPort;
    this.applications = applications;
    this.clientMode = clientMode;
    this.configuration = configuration;
    this.clusterSystemProperties = new ClusterSystemProperties(servers.size(), clientMode);
    this.numberOfGroups = numberOfGroups;
    this.builder = builder;
  }

  private String temporalStopMessage(Exception onError) {
    return "Failed to temporarily stop a server in Cluster Installation, reason: " + onError.getMessage();
  }

  private String restartMessage(Throwable onError) {
    return "Failed to restart servers in Cluster Installation, reason: " + onError.getMessage();
  }

  private void assertClusterStarted() {
    new PollingProber().check(new Probe() {

      @Override
      public boolean isSatisfied() {
        if (!clientMode) {
          HazelcastClusterManager hazelcastManager =
              servers.get(0).getCoreExtension(HazelcastClusterCoreExtension.class).getHazelcastManager();

          return hazelcastManager.getHazelcastInstance().getCluster().getMembers().size() == servers.size();
        }

        return true;
      }

      @Override
      public String describeFailure() {
        return "cluster was not created successfully";
      }
    });
  }

  public static class Builder extends AbstractInstallationBuilder<Builder> {

    Builder(int size, int numberOfGroups) {
      super(size, numberOfGroups);
    }

    @Override
    public FakeGatewayClusterInstallation build() {
      FakeGatewayInstallationConfiguration configuration =
          new FakeGatewayInstallationConfiguration(gatewayMode, false, null, gateKeeperMode, jdbcStoreConfiguration,
                                                   onApiDeleted);
      return new FakeGatewayClusterInstallation(applications, dynamicPort, configuration, clientModeEnabled, 1, this);
    }

    private List<FakeGatewayServer> buildServers() {
      return serverBuilders.stream()
          .map(FakeGatewayServerBuilder::build)
          .collect(toList());
    }
  }

}
