/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.tests.infrastructure.installation;

import static org.apache.commons.io.FileUtils.deleteQuietly;
import static org.junit.Assert.fail;
import static org.mule.runtime.core.internal.util.FunctionalUtils.safely;

import org.mule.runtime.api.exception.MuleException;

import com.mulesoft.anypoint.tests.infrastructure.FakeGatewayServer;
import com.mulesoft.anypoint.tests.infrastructure.rules.ClusterDynamicPort;
import com.mulesoft.anypoint.tita.environment.api.artifact.Artifact;

import java.io.IOException;
import java.util.List;

import org.junit.rules.ExternalResource;

/**
 * JUnit rule that set up a FakeGatewayServer and deploys the selected applications
 */
public class FakeGatewayInstallation extends ExternalResource implements Installation<FakeGatewayInstallation> {

  private Builder builder;
  private List<Artifact> applications;
  private List<Artifact> domains;
  private FakeGatewayInstallationConfiguration configuration;
  private FakeGatewayServer muleServer;

  public FakeGatewayInstallation(FakeGatewayServer muleServer, List<Artifact> applications,
                                 List<Artifact> domains,
                                 FakeGatewayInstallationConfiguration configuration, Builder builder) {
    this.muleServer = muleServer;
    this.applications = applications;
    this.domains = domains;
    this.configuration = configuration;
    this.builder = builder;
  }

  public static Builder builder() {
    return new Builder();
  }

  @Override
  protected void before() throws Throwable {
    try {
      muleServer.installDomains(domains);
      muleServer.installApplications(applications);
      spinUpServer();
    } catch (Throwable t) {
      after();
      throw t;
    }
  }

  @Override
  protected void after() {
    configuration.after();

    if (muleServer != null) {
      try {
        muleServer.stop();
      } catch (MuleException e) {
        throw new RuntimeException("Error trying to stop Fake Mule Server", e);
      } finally {
        deleteQuietly(muleServer.getMuleHome());
      }
    }
  }

  public FakeGatewayServer getServer() {
    return muleServer;
  }

  @Override
  public FakeGatewayInstallation restart() {
    safely(() -> {
      muleServer.stop();
      replicate(builder.build());
      spinUpServer();
    }, onError -> fail(restartMessage(onError)));
    return this;
  }

  @Override
  public FakeGatewayInstallation removePoliciesAndContext() {
    muleServer.removeAllPoliciesAndContext();
    return this;
  }

  public FakeGatewayInstallation updateHdpRegistry(String source) {
    return updateHdpRegistry(source, source);
  }

  public FakeGatewayInstallation updateHdpRegistry(String source, String target) {
    muleServer.updateHdpRegistry(source, target);
    return this;
  }

  private String restartMessage(Exception onError) {
    return "Failed to restart Standalone Installation, reason: " + onError.getMessage();
  }

  private void replicate(FakeGatewayInstallation newInstallation) {
    this.muleServer = newInstallation.muleServer;
    this.applications = newInstallation.applications;
    this.builder = newInstallation.builder;
  }

  private void spinUpServer() throws IOException, MuleException {
    muleServer.start();
    applications.forEach(app -> muleServer.assertDeploymentSuccess(app.getName()));
  }

  /**
   * Standalone installation builder
   */
  public static class Builder extends AbstractInstallationBuilder<Builder> {

    Builder() {
      super(1, 1);
    }

    @Override
    public FakeGatewayInstallation build() {
      FakeGatewayInstallationConfiguration configuration =
          new FakeGatewayInstallationConfiguration(gatewayMode, encrypted, encryptionKey, gateKeeperMode, onApiDeleted);

      // Set system properties before runtime classes are instantiated
      configuration.before();

      FakeGatewayServer muleServer = serverBuilders.get(0).build();

      return new FakeGatewayInstallation(muleServer, applications, domains, configuration, this);
    }

    @Override
    public Builder withDynamicPort(ClusterDynamicPort dynamicPort) {
      throw new IllegalStateException("Method available only for Cluster");
    }

  }
}
