/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.tests.infrastructure.rules;

import org.mule.runtime.api.lifecycle.Disposable;
import org.mule.tck.junit4.rule.FreePortFinder;

import com.mulesoft.anypoint.tests.infrastructure.FakeGatewayServer;

import java.util.HashMap;
import java.util.Map;

public class ClusterDynamicPort implements Disposable {

  private static final int DEFAULT_MIN_PORT = 10000;
  private static final int DEFAULT_MAX_PORT = 40000;

  private final FreePortFinder portFinder;
  private final String portName;
  private Map<FakeGatewayServer, Integer> serversAndPorts;

  public ClusterDynamicPort(String portName) {
    this(portName, DEFAULT_MIN_PORT, DEFAULT_MAX_PORT);
  }

  public ClusterDynamicPort(String portName, Integer minPortNumber, Integer maxPortNumber) {
    checkNotNull(portName, "Port name cannot be null");
    checkRange(minPortNumber, maxPortNumber);
    this.portName = portName;
    this.serversAndPorts = new HashMap<>();
    this.portFinder = new FreePortFinder(minPortNumber, maxPortNumber);
  }

  public Integer createPortForServer(FakeGatewayServer server) {
    checkNotAlreadyCreated(server);

    Integer newPort = portFinder.find();
    serversAndPorts.put(server, newPort);

    System.clearProperty(portName);
    System.setProperty(portName, Integer.toString(newPort));

    return newPort;
  }

  public Integer port(FakeGatewayServer server) {
    checkAlreadyCreated(server);
    return serversAndPorts.get(server);
  }

  public void setPortForServer(FakeGatewayServer server) {
    System.clearProperty(portName);
    System.setProperty(portName, Integer.toString(port(server)));
  }

  @Override
  public void dispose() {
    System.clearProperty(portName);
    for (FakeGatewayServer server : serversAndPorts.keySet()) {
      releasePort(server);
    }
    serversAndPorts.clear();
  }

  private void releasePort(FakeGatewayServer server) {
    int port = serversAndPorts.get(server);
    portFinder.releasePort(port);
  }

  private void checkRange(Integer minPortNumber, Integer maxPortNumber) {
    if (minPortNumber > maxPortNumber) {
      throw new Error("Min and Max port do not create a valid range");
    }
  }

  private void checkNotNull(String variable, String onErrorMessage) {
    if (variable == null) {
      throw new Error(onErrorMessage);
    }
  }

  private void checkNotAlreadyCreated(FakeGatewayServer server) {
    if (serversAndPorts.containsKey(server)) {
      throw new Error("Can only create one port for application: " + server);
    }
  }

  private void checkAlreadyCreated(FakeGatewayServer server) {
    if (!serversAndPorts.containsKey(server)) {
      throw new Error("No port has been created for application: " + server);
    }
  }
}
