/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.tests.infrastructure;

import static com.mulesoft.anypoint.tests.GatewayMatchers.isInRange;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.tck.junit4.rule.FreePortFinder;

import com.mulesoft.anypoint.tests.infrastructure.rules.ClusterDynamicPort;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

public class ClusterDynamicPortTests {

  private static final Integer min = 8000;
  private static final Integer max = 10000;
  private static final String PORT = "port";
  @Rule
  public ExpectedException thrown = ExpectedException.none();
  private FakeGatewayServer server;
  private FakeGatewayServer anotherServer;

  @Before
  public void setUp() {
    server = mockedServer("server1");
    anotherServer = mockedServer("server2");
  }

  @Test
  public void portNameCannotBeNull() {
    thrown.expectMessage("Port name cannot be null");

    new ClusterDynamicPort(null, 1, 2);
  }

  @Test
  public void portRangeMustBeARange() {
    thrown.expectMessage("Min and Max port do not create a valid range");

    new ClusterDynamicPort("port", 10, 3);
  }

  @Test
  public void createPortGiveAFreePort() {
    ClusterDynamicPort dynamicPort = new ClusterDynamicPort(PORT, min, max);
    Integer createdPort = dynamicPort.createPortForServer(server);
    FreePortFinder portFinder = new FreePortFinder(min, max);

    assertThat(createdPort, isInRange(min, max));
    assertTrue("Port is not free!", portFinder.isPortFree(createdPort));
    assertEquals("Creating and asking for the port should return the same", createdPort, dynamicPort.port(server));
  }

  @Test
  public void creatingPortForAppMoreThanOnceRaisesException() {
    thrown.expectMessage("Can only create one port for application: server1");
    ClusterDynamicPort dynamicPort = new ClusterDynamicPort(PORT, min, max);
    dynamicPort.createPortForServer(server);
    dynamicPort.createPortForServer(server);
  }

  @Test
  public void toRequestThePortFirstItMustBeCreated() {
    thrown.expectMessage("No port has been created for application: server1");
    new ClusterDynamicPort(PORT, min, max).port(server);
  }

  @Test
  public void systemPropertyWithPortNameGetsSet() {
    Integer port = new ClusterDynamicPort(PORT, min, max).createPortForServer(server);
    assertNotNull(System.getProperty(PORT));
    assertEquals(Integer.toString(port), System.getProperty(PORT));
  }

  @Test
  public void whenDisposeSystemPropertyIsCleared() {
    ClusterDynamicPort dynamicPort = new ClusterDynamicPort(PORT, min, max);
    dynamicPort.createPortForServer(server);
    dynamicPort.dispose();
    assertNull(System.getProperty(PORT));
  }

  @Test
  public void canCreateForSeveralServers() {
    ClusterDynamicPort dynamicPort = new ClusterDynamicPort(PORT, min, max);
    Integer serverPort = dynamicPort.createPortForServer(server);
    Integer anotherServerPort = dynamicPort.createPortForServer(anotherServer);
    assertNotEquals(serverPort, anotherServerPort);
  }

  private FakeGatewayServer mockedServer(String name) {
    FakeGatewayServer fakeServer = mock(FakeGatewayServer.class);
    when(fakeServer.toString()).thenReturn(name);
    return fakeServer;
  }
}
