/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.test.policy.offline;

import static com.mulesoft.anypoint.test.policy.offline.GatewayVersionFilter.TITA_TESTING_VALUE;
import static com.mulesoft.anypoint.tita.environment.api.artifact.Identifier.identifier;
import static com.mulesoft.mule.runtime.gw.api.config.GatewaySecurityConfiguration.EncryptionMode.FULL;
import static com.mulesoft.mule.runtime.gw.api.config.GatewaySecurityConfiguration.EncryptionMode.SENSITIVE_ONLY;
import static java.util.Arrays.asList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertEquals;

import com.mulesoft.anypoint.tests.http.HttpRequest;
import com.mulesoft.anypoint.tita.environment.api.ApplicationSelector;
import com.mulesoft.anypoint.tita.environment.api.anypoint.ApiManager;
import com.mulesoft.anypoint.tita.environment.api.artifact.ApplicationBuilder;
import com.mulesoft.anypoint.tita.environment.api.artifact.Identifier;
import com.mulesoft.anypoint.tita.environment.api.artifact.PolicyArtifact;
import com.mulesoft.anypoint.tita.environment.api.artifact.policy.PolicySupplier;
import com.mulesoft.anypoint.tita.environment.api.runtime.Runtime;
import com.mulesoft.anypoint.tita.environment.api.runtime.builder.RuntimeConfiguration;
import com.mulesoft.anypoint.tita.environment.api.validator.EncryptedFileValidator;
import com.mulesoft.anypoint.tita.environment.api.validator.EncryptionMatcherByList;
import com.mulesoft.anypoint.tita.runner.ambar.Ambar;
import com.mulesoft.anypoint.tita.runner.ambar.annotation.Application;
import com.mulesoft.anypoint.tita.runner.ambar.annotation.Platform;
import com.mulesoft.anypoint.tita.runner.ambar.annotation.Policy;
import com.mulesoft.anypoint.tita.runner.ambar.annotation.TestTarget;
import com.mulesoft.anypoint.tita.runner.ambar.annotation.runtime.Encryption;
import com.mulesoft.anypoint.tita.runner.ambar.annotation.runtime.Standalone;

import java.util.HashMap;
import java.util.Map;

import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.runner.RunWith;

@RunWith(Ambar.class)
public class PolicyDeploymentUsingFullEncryptionTestCase {

  private static final String ENCRYPTION_KEY = "GatewayTeamKey00";

  private static final String CONFIG_PROPERTY_SMART_KEY = "smartKey";
  private static final String CONFIG_PROPERTY_DUMB_KEY = "dumbKey";
  private static final String CONFIG_PROPERTY_LIST_KEY = "listKey";
  private static final String CONFIG_PROPERTY_EXPRESSION_KEY = "expression";

  private static final String PAYLOAD_BEFORE_ENCRYPTION = "Oh no, you have desencrypted me!";
  private static final String CONFIG_PROPERTY_BOOLEAN_VALUE = "true";
  private static final String CONFIG_PROPERTY_INT_VALUE = "1234";
  private static final String CONFIG_PROPERTY_IDENTITY_MANAGEMENT = "identityManagementTokenUrl";
  private static final String VALUE_IDENTITY_MANAGEMENT = "http://hello.gateway.com";

  private static final String LIST_VALUE_1 = "listKey1";
  private static final String LIST_VALUE_2 = "listKey2";

  private static Identifier api = identifier("api.id1");
  private static Identifier port = identifier("port");

  @Standalone(testing = TITA_TESTING_VALUE, encryption = @Encryption(mode = FULL, key = ENCRYPTION_KEY))
  private Runtime runtime;

  @Platform
  protected ApiManager apiManager;

  @TestTarget
  @Policy(groupId = "com.mulesoft.anypoint.tita.policy", artifactId = "smart-encryption-policy")
  private PolicySupplier smartEncryptionPolicy;

  @Application
  public static ApplicationBuilder app(ApplicationSelector selector) {
    return selector
        .custom("customAppFull", "mule-config-http.xml")
        .withApi(api, port);
  }

  private HttpRequest request;
  private PolicyArtifact policyArtifact;

  @Before
  public void setUpTest() {
    request = runtime.api(api).request("/api/empty");
  }

  private Map<String, Object> policyConfiguration() {
    Map<String, Object> properties = basePolicyConfiguration();
    properties.put(CONFIG_PROPERTY_EXPRESSION_KEY, PAYLOAD_BEFORE_ENCRYPTION);
    return properties;
  }

  private Map<String, Object> escapedKeyPolicyConfiguration() {
    Map<String, Object> properties = basePolicyConfiguration();
    String escapedKey = "&quot;__&lt;Guns_&amp;&amp;_Roses&#39;&gt;__&quot;";
    properties.put(CONFIG_PROPERTY_EXPRESSION_KEY, escapedKey);
    return properties;
  }

  private Map<String, Object> basePolicyConfiguration() {
    Map<String, Object> properties = new HashMap<>();
    properties.put(CONFIG_PROPERTY_SMART_KEY, CONFIG_PROPERTY_BOOLEAN_VALUE);
    properties.put(CONFIG_PROPERTY_DUMB_KEY, CONFIG_PROPERTY_INT_VALUE);
    properties.put(CONFIG_PROPERTY_LIST_KEY, asList(LIST_VALUE_1, LIST_VALUE_2));
    properties.put(CONFIG_PROPERTY_IDENTITY_MANAGEMENT, VALUE_IDENTITY_MANAGEMENT);
    return properties;
  }

  @Test
  public void deploySinglePolicyUsingEncryption() {
    policyArtifact = apiManager.apply(api, smartEncryptionPolicy, policyConfiguration());

    assertPolicyFullEncrypted();

    assertEquals("Policy not correctly applied",
                 CONFIG_PROPERTY_BOOLEAN_VALUE + CONFIG_PROPERTY_INT_VALUE + PAYLOAD_BEFORE_ENCRYPTION, request.get().asString());
  }

  @Test
  public void restartingGatewayShouldDeploySuccessfully() {
    policyArtifact = apiManager.apply(api, smartEncryptionPolicy, policyConfiguration());

    assertPolicyFullEncrypted();

    assertEquals("Policy not correctly applied",
                 CONFIG_PROPERTY_BOOLEAN_VALUE + CONFIG_PROPERTY_INT_VALUE + PAYLOAD_BEFORE_ENCRYPTION, request.get().asString());

    runtime.restart();

    assertEquals("Policy not correctly applied",
                 CONFIG_PROPERTY_BOOLEAN_VALUE + CONFIG_PROPERTY_INT_VALUE + PAYLOAD_BEFORE_ENCRYPTION, request.get().asString());

    assertPolicyFullEncrypted();
  }

  @Test
  public void restartingUsingSmartGatewayShouldDeploySuccessfully() {
    policyArtifact = apiManager.apply(api, smartEncryptionPolicy, policyConfiguration());

    assertPolicyFullEncrypted();

    assertEquals("Policy not correctly applied",
                 CONFIG_PROPERTY_BOOLEAN_VALUE + CONFIG_PROPERTY_INT_VALUE + PAYLOAD_BEFORE_ENCRYPTION, request.get().asString());

    runtime.restart(smartEncryptionConfig());

    assertEquals("Policy not correctly applied",
                 CONFIG_PROPERTY_BOOLEAN_VALUE + CONFIG_PROPERTY_INT_VALUE + PAYLOAD_BEFORE_ENCRYPTION, request.get().asString());

    assertPolicySensitiveOnlyEncrypted();

  }

  @Ignore("AGW-4657: Allow assertion of failed policies")
  @Test
  public void restartingGatewayWithPreviousConfigurationAndNoKeyShouldGiveAnError() {
    policyArtifact = apiManager.apply(api, smartEncryptionPolicy, policyConfiguration());

    assertPolicyFullEncrypted();

    assertEquals("Policy not correctly applied",
                 CONFIG_PROPERTY_BOOLEAN_VALUE + CONFIG_PROPERTY_INT_VALUE + PAYLOAD_BEFORE_ENCRYPTION, request.get().asString());

    runtime.restart(noEncryptionConfig());

    // check(() -> assertThat(tita.runtime().fileSystem().policyFailures(), hasSize(1)));
    // assertThat(request.get().asString(), isEmptyString());
  }

  @Ignore("AGW-4657: Allow assertion of failed policies")
  @Test
  public void restartingGatewayWithPreviousConfigurationAndEncryptedPropertiesFileIsDeleted() {
    policyArtifact = apiManager.apply(api, smartEncryptionPolicy, policyConfiguration());

    assertPolicyFullEncrypted();

    assertEquals("Policy not correctly applied",
                 CONFIG_PROPERTY_BOOLEAN_VALUE + CONFIG_PROPERTY_INT_VALUE + PAYLOAD_BEFORE_ENCRYPTION, request.get().asString());

    runtime.restart(noEncryptionConfig());

    assertThat(runtime.state().policyIsNotEncrypted(policyArtifact.id()), is(true));
  }

  @Test
  public void encryptionOfEscapedCharactersArePreviouslyUnescaped() {
    String unescapedKey = "\"__<Guns_&&_Roses\'>__\"";
    policyArtifact = apiManager.apply(api, smartEncryptionPolicy, escapedKeyPolicyConfiguration());

    assertPolicyFullEncrypted();

    assertEquals("Policy not correctly applied",
                 CONFIG_PROPERTY_BOOLEAN_VALUE + CONFIG_PROPERTY_INT_VALUE + unescapedKey, request.get().asString());

  }

  private void assertPolicyFullEncrypted() {
    assertThat(runtime.state().isPolicyEncrypted(policyArtifact.id(), fullListMatcher()), is(true));
  }

  private void assertPolicySensitiveOnlyEncrypted() {
    assertThat(runtime.state()
        .isPolicyEncrypted(policyArtifact.id(),
                           new EncryptionMatcherByList(asList(CONFIG_PROPERTY_SMART_KEY, CONFIG_PROPERTY_IDENTITY_MANAGEMENT))),
               is(true));
    assertThat(runtime.state().isPolicyEncrypted(policyArtifact.id(),
                                                 new EncryptionMatcherByList(asList(CONFIG_PROPERTY_DUMB_KEY,
                                                                                    CONFIG_PROPERTY_EXPRESSION_KEY))),
               is(false));
  }

  private EncryptedFileValidator fullListMatcher() {
    return new EncryptionMatcherByList(asList(CONFIG_PROPERTY_SMART_KEY, CONFIG_PROPERTY_DUMB_KEY, CONFIG_PROPERTY_EXPRESSION_KEY,
                                              CONFIG_PROPERTY_IDENTITY_MANAGEMENT));
  }

  private RuntimeConfiguration noEncryptionConfig() {
    return RuntimeConfiguration.builder()
        .build();
  }

  private RuntimeConfiguration smartEncryptionConfig() {
    return RuntimeConfiguration.builder()
        .withEncryption(SENSITIVE_ONLY, ENCRYPTION_KEY)
        .build();
  }

}
