/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.test.policy.offline;

import static com.mulesoft.anypoint.test.policy.offline.GatewayVersionFilter.TITA_TESTING_VALUE;
import static com.mulesoft.anypoint.tita.environment.api.artifact.Identifier.identifier;
import static com.mulesoft.mule.runtime.gw.api.config.GatewaySecurityConfiguration.EncryptionMode.FULL;
import static com.mulesoft.mule.runtime.gw.api.config.GatewaySecurityConfiguration.EncryptionMode.SENSITIVE_ONLY;
import static java.util.Arrays.asList;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;

import com.mulesoft.anypoint.tests.http.HttpRequest;
import com.mulesoft.anypoint.tests.http.HttpResponse;
import com.mulesoft.anypoint.tita.environment.api.ApplicationSelector;
import com.mulesoft.anypoint.tita.environment.api.anypoint.ApiManager;
import com.mulesoft.anypoint.tita.environment.api.artifact.ApplicationBuilder;
import com.mulesoft.anypoint.tita.environment.api.artifact.Identifier;
import com.mulesoft.anypoint.tita.environment.api.artifact.PolicyArtifact;
import com.mulesoft.anypoint.tita.environment.api.artifact.policy.PolicySupplier;
import com.mulesoft.anypoint.tita.environment.api.runtime.Runtime;
import com.mulesoft.anypoint.tita.environment.api.runtime.builder.RuntimeConfiguration;
import com.mulesoft.anypoint.tita.environment.api.validator.EncryptionMatcherByList;
import com.mulesoft.anypoint.tita.runner.ambar.Ambar;
import com.mulesoft.anypoint.tita.runner.ambar.annotation.Application;
import com.mulesoft.anypoint.tita.runner.ambar.annotation.Platform;
import com.mulesoft.anypoint.tita.runner.ambar.annotation.Policy;
import com.mulesoft.anypoint.tita.runner.ambar.annotation.TestTarget;
import com.mulesoft.anypoint.tita.runner.ambar.annotation.runtime.Encryption;
import com.mulesoft.anypoint.tita.runner.ambar.annotation.runtime.Standalone;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.Map;

import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.runner.RunWith;

@RunWith(Ambar.class)
public class PolicyDeploymentUsingSmartEncryptionTestCase {

  private static final String ENCRYPTION_KEY = "GatewayTeamKey00";

  private static final String CONFIG_PROPERTY_SMART_KEY = "smartKey";
  private static final String CONFIG_PROPERTY_DUMB_KEY = "dumbKey";
  private static final String CONFIG_PROPERTY_LIST_KEY = "listKey";
  private static final String CONFIG_PROPERTY_EXPRESSION_KEY = "expression";
  private static final String CONFIG_PROPERTY_IDENTITY_MANAGEMENT = "identityManagementTokenUrl";

  private static final String VALUE_SMART = "Hello";
  private static final String VALUE_DUMB = "World";
  private static final String VALUE_IDENTITY_MANAGEMENT = "http://hello.gateway.com";

  private static final String HEADER_KEY = "x-test";
  private static final String HEADER_VALUE = "Encoded";
  private static final String VALUE_EXPRESSION_ENCRYPTED = "#[attributes.headers[&#39;" + HEADER_KEY + "&#39;]]";

  private static Identifier api = identifier("api.id1");
  private static Identifier port = identifier("port");

  public static final String EXPECTED_PAYLOAD = VALUE_SMART + VALUE_DUMB + HEADER_VALUE;

  @Standalone(testing = TITA_TESTING_VALUE, encryption = @Encryption(mode = SENSITIVE_ONLY, key = ENCRYPTION_KEY))
  private Runtime runtime;

  @Platform
  protected ApiManager apiManager;

  @TestTarget
  @Policy(groupId = "com.mulesoft.anypoint.tita.policy", artifactId = "smart-encryption-policy")
  private PolicySupplier smartEncryptionPolicy;

  @Policy(groupId = "com.mulesoft.anypoint.tita.policy", artifactId = "concatenate-keys-policy")
  private PolicySupplier concatenateKeysPolicy;

  @Application
  public static ApplicationBuilder app(ApplicationSelector selector) {
    return selector
        .custom("customAppSmart", "mule-config-http.xml")
        .withApi(api, port);
  }

  private HttpRequest request;
  private PolicyArtifact policyArtifact;

  @Before
  public void setUpTest() {
    request = runtime.api(api).request("/api/empty").withHeader(HEADER_KEY, HEADER_VALUE);
  }

  @Test
  public void deploySinglePolicyUsingSmartEncryption() {
    policyArtifact = apiManager.apply(api, smartEncryptionPolicy, policyConfiguration());

    assertPolicySensitiveOnlyEncrypted();

    HttpResponse response = request.get();

    assertEquals("Policy not correctly applied", EXPECTED_PAYLOAD, response.asString());
  }

  @Test
  public void policyWithoutEncryptedValues() {
    PolicyArtifact policyArtifactWithoutEncryptedValues = apiManager.apply(api, concatenateKeysPolicy, policyConfiguration());

    assertThat(runtime.state().isPolicyEncrypted(policyArtifactWithoutEncryptedValues.id(),
                                                 new EncryptionMatcherByList(asList(CONFIG_PROPERTY_SMART_KEY,
                                                                                    CONFIG_PROPERTY_DUMB_KEY,
                                                                                    CONFIG_PROPERTY_LIST_KEY,
                                                                                    CONFIG_PROPERTY_EXPRESSION_KEY,
                                                                                    CONFIG_PROPERTY_IDENTITY_MANAGEMENT))),
               is(false));
    HttpResponse response = request.get();

    assertThat(response.statusCode(), is(200));
    assertEquals("Policy not correctly applied", EXPECTED_PAYLOAD, response.asString());
  }

  @Test
  @Ignore("AGW-4657: Allow assertion of failed policies")
  public void restartingGatewayWithPreviousConfigurationAndNoKeyShouldGiveAnError() throws IOException, URISyntaxException {
    policyArtifact = apiManager.apply(api, smartEncryptionPolicy, policyConfiguration());

    assertPolicySensitiveOnlyEncrypted();

    assertEquals("Policy not correctly applied", EXPECTED_PAYLOAD, request.get().asString());

    runtime.restart(noEncryptionConfig());

    assertEquals("Policy not correctly applied", EXPECTED_PAYLOAD, request.get().asString());
    // check(() -> assertThat(tita.runtime().fileSystem().policyFailures(), hasSize(1)));
    // assertThat(request.get().asString(), isEmptyStringw());
  }

  @Test
  public void restartingGatewayShouldDeploySuccessfully() {
    policyArtifact = apiManager.apply(api, smartEncryptionPolicy, policyConfiguration());

    assertPolicySensitiveOnlyEncrypted();

    assertEquals("Policy not correctly applied", EXPECTED_PAYLOAD, request.get().asString());

    runtime.restart();

    assertPolicySensitiveOnlyEncrypted();

    assertEquals("Policy not correctly applied", EXPECTED_PAYLOAD, request.get().asString());
  }

  @Test
  public void changingFromSmartEncryptionToFullEncryption() {
    policyArtifact = apiManager.apply(api, smartEncryptionPolicy, policyConfiguration());

    assertPolicySensitiveOnlyEncrypted();

    assertEquals("Policy not correctly applied", EXPECTED_PAYLOAD, request.get().asString());

    runtime.restart(fullEncryptionConfiguration());

    assertThat(runtime.state().isPolicyEncrypted(policyArtifact.id(),
                                                 new EncryptionMatcherByList(asList(
                                                                                    CONFIG_PROPERTY_SMART_KEY,
                                                                                    CONFIG_PROPERTY_DUMB_KEY,
                                                                                    CONFIG_PROPERTY_EXPRESSION_KEY,
                                                                                    CONFIG_PROPERTY_IDENTITY_MANAGEMENT))),
               is(true));

    assertEquals("Policy not correctly applied", EXPECTED_PAYLOAD, request.get().asString());
  }

  private Map<String, Object> policyConfiguration() {
    Map<String, Object> properties = new HashMap<>();
    properties.put(CONFIG_PROPERTY_SMART_KEY, VALUE_SMART);
    properties.put(CONFIG_PROPERTY_DUMB_KEY, VALUE_DUMB);
    properties.put(CONFIG_PROPERTY_LIST_KEY, asList(VALUE_SMART, VALUE_DUMB));
    properties.put(CONFIG_PROPERTY_EXPRESSION_KEY, VALUE_EXPRESSION_ENCRYPTED);
    properties.put(CONFIG_PROPERTY_IDENTITY_MANAGEMENT, VALUE_IDENTITY_MANAGEMENT);
    return properties;
  }

  private RuntimeConfiguration fullEncryptionConfiguration() {
    return new RuntimeConfiguration.Builder()
        .withEncryption(FULL, ENCRYPTION_KEY)
        .gateKeeperDisabled()
        .build();
  }

  private RuntimeConfiguration noEncryptionConfig() {
    return RuntimeConfiguration.builder()
        .build();
  }

  private void assertPolicySensitiveOnlyEncrypted() {
    assertThat(runtime.state().isPolicyEncrypted(policyArtifact.id(),
                                                 new EncryptionMatcherByList(asList(CONFIG_PROPERTY_SMART_KEY,
                                                                                    CONFIG_PROPERTY_IDENTITY_MANAGEMENT))),
               is(true));
    assertThat(runtime.state()
        .isPolicyEncrypted(policyArtifact.id(),
                           new EncryptionMatcherByList(asList(CONFIG_PROPERTY_DUMB_KEY, CONFIG_PROPERTY_EXPRESSION_KEY))),
               is(false));
  }

}
