/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.test.policy.zero.downtime;

import static com.mulesoft.anypoint.test.policy.GatewayVersionFilter.TITA_TESTING_VALUE;
import static com.mulesoft.anypoint.tita.environment.api.Tita.performanceMatchers;
import static com.mulesoft.anypoint.tita.environment.api.artifact.Identifier.identifier;
import static java.util.Collections.singletonList;
import static java.util.Collections.singletonMap;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.fail;

import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.probe.PollingProber;

import com.mulesoft.anypoint.tests.DescriptiveProbe;
import com.mulesoft.anypoint.tests.http.HttpRequest;
import com.mulesoft.anypoint.tita.environment.api.ApplicationSelector;
import com.mulesoft.anypoint.tita.environment.api.anypoint.ApiManager;
import com.mulesoft.anypoint.tita.environment.api.artifact.ApplicationBuilder;
import com.mulesoft.anypoint.tita.environment.api.artifact.Identifier;
import com.mulesoft.anypoint.tita.environment.api.artifact.PolicyArtifact;
import com.mulesoft.anypoint.tita.environment.api.artifact.policy.PolicySupplier;
import com.mulesoft.anypoint.tita.environment.api.performance.PerformanceEngine;
import com.mulesoft.anypoint.tita.environment.api.performance.PerformanceResult;
import com.mulesoft.anypoint.tita.environment.api.performance.StatusCode;
import com.mulesoft.anypoint.tita.environment.api.runtime.Runtime;
import com.mulesoft.anypoint.tita.environment.api.runtime.builder.RuntimeConfiguration;
import com.mulesoft.anypoint.tita.runner.ambar.Ambar;
import com.mulesoft.anypoint.tita.runner.ambar.annotation.Application;
import com.mulesoft.anypoint.tita.runner.ambar.annotation.Performance;
import com.mulesoft.anypoint.tita.runner.ambar.annotation.Platform;
import com.mulesoft.anypoint.tita.runner.ambar.annotation.Policy;
import com.mulesoft.anypoint.tita.runner.ambar.annotation.TestTarget;
import com.mulesoft.anypoint.tita.runner.ambar.annotation.runtime.Property;
import com.mulesoft.anypoint.tita.runner.ambar.annotation.runtime.Standalone;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

import org.junit.Ignore;
import org.junit.Test;
import org.junit.runner.RunWith;

@RunWith(Ambar.class)
public class ReorderTestCase extends AbstractMuleTestCase {

  protected static Identifier api = identifier("apiId");
  protected static Identifier port = identifier("port");

  @Standalone(testing = TITA_TESTING_VALUE, properties = {@Property(key = "throttling.persistence_enabled", value = "false")})
  protected Runtime runtime;

  @Platform
  protected ApiManager apiManager;

  @Application
  public static ApplicationBuilder app(ApplicationSelector selector) {
    return selector
        .custom("customAppZeroDowntime", "zeroDowntime.xml")
        .withApi(api, port);
  }

  @TestTarget
  @Policy(artifactId = "rate-limiting")
  private PolicySupplier policy;

  @Policy(artifactId = "config-failure-policy")
  private PolicySupplier payloadPolicy;

  @Performance
  PerformanceEngine engine;

  @Test
  public void successfulReorder() {

    Map<String, Object> reorderedPolicyConfig = failureConfig(false, "#['policy1' ++ payload]");

    PolicyArtifact artifact = apiManager.apply(api, payloadPolicy, 1, reorderedPolicyConfig);
    apiManager.apply(api, payloadPolicy, 2, failureConfig(false, "#['policy2' ++ payload]"));

    assertThat(request().get().asString(), is("policy1policy2"));

    apiManager.reorder(artifact, 3);

    assertThat(request().get().asString(), is("policy2policy1"));
  }

  @Test
  public void internalStatusPreservedOnReorder() {
    PolicyArtifact artifact = apiManager.apply(api, policy, 1, rateLimitConfig("#['id1']"));

    assertThat(request().get().statusCode(), is(200));

    apiManager.reorder(artifact, 2);

    assertThat(request().get().statusCode(), is(429));
  }

  @Test
  public void noInsecurityWindowOnReorder() throws ExecutionException, InterruptedException {
    PolicyArtifact artifact = apiManager.apply(api, policy, 1, rateLimitConfig("#['id1']"));

    assertThat(request().get().statusCode(), is(200));

    Future<PerformanceResult> result = engine
        .request(this::request)
        .withThreads(10)
        .during(10000)
        .runAsync();

    apiManager.reorder(artifact, 2);

    assertThat(result.get(), performanceMatchers().allResponsesAre(singletonList(new StatusCode(429))));
  }

  @Test
  public void reorderPreservedOnRestart() {

    Map<String, Object> reorderedPolicyConfig = failureConfig(false, "#['policy1' ++ payload]");

    PolicyArtifact artifact = apiManager.apply(api, payloadPolicy, 1, reorderedPolicyConfig);
    apiManager.apply(api, payloadPolicy, 2, failureConfig(false, "#['policy2' ++ payload]"));

    assertThat(request().get().asString(), is("policy1policy2"));

    apiManager.reorder(artifact, 3);

    assertThat(request().get().asString(), is("policy2policy1"));

    runtime.restart(RuntimeConfiguration.builder().gateKeeperDisabled().offline().build());

    assertThat(request().get().asString(), is("policy2policy1"));
  }

  protected HttpRequest request() {
    return runtime.api(api).request("api/main/200");
  }

  protected Map<String, Object> failureConfig(Boolean fail, String payload) {
    Map<String, Object> configData = new HashMap<>();
    configData.put("text", payload);
    configData.put("fail", fail.toString());

    return configData;
  }

  protected Map<String, Object> rateLimitConfig(String key) {
    Map<String, Object> configData = new HashMap<>();
    Map<String, Object> rateLimits = new HashMap<>();

    configData.put("keySelector", key);
    configData.put("clusterizable", false);

    rateLimits.put("timePeriodInMilliseconds", 100000);
    rateLimits.put("maximumRequests", 1);

    configData.put("rateLimits", singletonList(rateLimits));
    return configData;
  }

  private void probe(Runnable runnable) {
    new PollingProber(3000, 100).check(new DescriptiveProbe(runnable));
  }
}
