/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.test.policy.deployment;

import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.anypoint.tests.http.ApacheHttpRequest.request;
import static com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation.builder;
import static com.mulesoft.anypoint.tita.TestDependencies.testAssertionsDependency;
import static com.mulesoft.anypoint.tita.TestDependencies.testAuthenticationDependency;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactFileFactory.HTTP_DEPENDENCY;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestApplication;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestPolicyTemplate;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.startsWith;
import static org.hamcrest.core.IsNot.not;

import org.mule.functional.util.http.HttpMessage;

import com.mulesoft.anypoint.tests.http.HttpResponse;
import com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;

import org.junit.After;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.RuleChain;

public class FailingRequesterTestCase extends ExternalServerConnectionTestCase {

  protected static final ApiKey NULL_HEADER_VALUE = new ApiKey(1L);
  protected static final ApiKey NULL_HEADER_KEY = new ApiKey(2L);
  protected static final ApiKey NULL_HEADERS = new ApiKey(3L);
  protected static final ApiKey NULL_QUERY_PARAMS = new ApiKey(4L);
  protected static final ApiKey NULL_QUERY_PARAM_VALUE = new ApiKey(5L);
  protected static final ApiKey NULL_URI_PARAMS = new ApiKey(6L);
  protected static final ApiKey NULL_URI_PARAM_VALUE = new ApiKey(7L);
  protected static final ApiKey NULL_PATH = new ApiKey(8L);

  private static FakeGatewayInstallation installation =
      builder()
          .withApplications(buildTestApplication(APP, "mule-config-http-with-proxy-with-errors.xml", testAssertionsDependency(),
                                                 testAuthenticationDependency()))
          .withPolicyTemplates(buildTestPolicyTemplate(POLICY_TEMPLATE_KEY, "policies/policy-operation-append-payload.xml",
                                                       HTTP_DEPENDENCY))
          .gateKeeperDisabled()
          .offline()
          .build();

  @ClassRule
  public static RuleChain ruleChain = RuleChain.outerRule(port)
      .around(externalServer)
      .around(installation);

  @Test
  public void headerValueFixedAsNull() {
    deploy(NULL_HEADER_VALUE);
    HttpResponse response = get("failing-header-value");

    HttpMessage externalRequest = externalServer.getHttpServer().getLastHttpRequest();
    assertThat(response.statusCode(), is(200));
    assertThat(externalRequest, notNullValue());
    assertThat(externalRequest.getHeaders().keySet(), not(contains("somenullheader")));
    assertNoPolicy(response);
  }

  @Test
  public void headerKeyAsNull() {
    deploy(NULL_HEADER_KEY);
    HttpResponse response = get("failing-header-key");

    assertThat(response.statusCode(), is(500));
    assertThat(response.asString(), startsWith("\"Script '{vars.'garlopa': 'someHeaderValue'} ' has errors:"));
  }

  @Test
  public void wholeHeaderResolutionIsNull() {
    deploy(NULL_HEADERS);
    HttpResponse response = get("null-headers");

    HttpMessage externalRequest = externalServer.getHttpServer().getLastHttpRequest();
    assertThat(response.statusCode(), is(200));
    assertThat(externalRequest, notNullValue());
    assertNoPolicy(response);
  }

  @Test
  public void nullQueryParams() {
    deploy(NULL_QUERY_PARAMS);
    HttpResponse response = get("null-query-params");

    assertThat(response.statusCode(), is(500));
    assertThat(response.asString(), is("unknown description"));
  }

  @Test
  public void nullQueryParamValue() {
    deploy(NULL_QUERY_PARAM_VALUE);
    HttpResponse response = get("null-query-param-value");

    HttpMessage externalRequest = externalServer.getHttpServer().getLastHttpRequest();
    assertThat(response.statusCode(), is(200));
    assertThat(externalRequest, notNullValue());
    assertNoPolicy(response);
  }

  @Test
  public void nullUriParams() {
    deploy(NULL_URI_PARAMS);
    HttpResponse response = get("null-uri-params");

    assertThat(response.statusCode(), is(500));
    assertThat(response.asString(), is("unknown description"));
  }

  @Test
  public void nullUriParamValue() {
    deploy(NULL_URI_PARAM_VALUE);
    HttpResponse response = get("null-uri-param-value");

    assertThat(response.statusCode(), is(500));
    assertThat(response.asString(), is("Expression {flowUriParam} evaluated to null."));
  }

  @Test
  public void nullPath() {
    deploy(NULL_PATH);
    HttpResponse response = get("null-path");

    assertThat(response.statusCode(), is(500));
    assertThat(response.asString(),
               is("Required parameter 'path' was assigned with value '#[vars.'garlopa']' which resolved to null. Required parameters need to be assigned with non null values"));
  }

  private HttpResponse get(String endpoint) {
    return request(port, "api/" + endpoint).get();
  }

  private void deploy(ApiKey apiKey) {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY, apiKey);

    installation.getServer().deployPolicy(policyDefinition);
  }

  @After
  public void tearDown() {
    installation.getServer().removeAllPoliciesAndContext();
  }

  private void assertNoPolicy(HttpResponse response) {
    assertThat(response.asString(), is("ExternalServerPayload"));
  }
}
