/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.test.policy.deployment;

import static com.google.common.collect.ImmutableMap.of;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_PAYLOAD;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_PAYLOAD_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY_3;
import static com.mulesoft.anypoint.tests.http.ApacheHttpRequest.request;
import static com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation.builder;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestApplication;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestPolicyTemplate;
import static com.mulesoft.mule.runtime.gw.policies.store.DefaultPolicyStore.POLICY_CONFIG_XML_FILE_NAME;
import static com.mulesoft.mule.runtime.gw.policies.store.DefaultPolicyStore.POLICY_DEFINITION_JSON_FILE_NAME;
import static java.util.Collections.emptyMap;
import static org.hamcrest.Matchers.arrayWithSize;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.hasItems;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;

import org.mule.runtime.deployment.model.api.policy.PolicyTemplateDescriptor;

import com.mulesoft.anypoint.tests.http.HttpRequest;
import com.mulesoft.anypoint.tests.infrastructure.FakeGatewayServer;
import com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation;
import com.mulesoft.mule.runtime.gw.api.policy.PolicyTemplateKey;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.model.ValidPolicySpecification;
import com.mulesoft.mule.runtime.gw.policies.Policy;
import com.mulesoft.mule.runtime.gw.policies.template.PolicyTemplate;

import java.io.File;

import org.junit.After;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.RuleChain;

public class PolicyDeploymentScenariosTestCase extends PolicyDeploymentTestCase {

  private static FakeGatewayInstallation installation;

  @ClassRule
  public static RuleChain chain = RuleChain.outerRule(httpPort)
      .around(policyPayload)
      .around(policyPayload2)
      .around(installation());

  private FakeGatewayServer server = installation.getServer();

  @After
  public void tearDown() {
    server.removeAllPoliciesAndContext();
  }

  @Test
  public void deploySinglePolicy() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY);

    server.deployPolicy(policyDefinition);

    assertEquals("Policy not correctly applied", POLICY_PAYLOAD, request.get().asString());
    assertThat(policyStore.load(), hasSize(1));
    assertThat(policyStore.load().get(0), is(policyDefinition));

    checkPhysicalFiles(policyDefinition);
  }

  @Test
  public void deployASecondPolicy() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY_2, 2);
    PolicyDefinition policyDefinition2 = policyDefinition(POLICY_ID_2, POLICY_TEMPLATE_KEY_2, 1);

    server.deployPolicy(policyDefinition);

    assertEquals("Policy not correctly applied", POLICY_PAYLOAD_2, request.get().asString());

    server.deployPolicy(policyDefinition2);

    assertEquals("Second Policy not correctly applied", POLICY_PAYLOAD_2 + POLICY_PAYLOAD_2,
                 request.get().asString());

    assertThat(policyStore.load(), hasSize(2));
    assertThat(policyStore.load(), hasItems(policyDefinition, policyDefinition2));

    checkPhysicalFiles(policyDefinition);
    checkPhysicalFiles(policyDefinition2);
  }

  @Test
  public void deployASecondPolicyWithDifferentTemplates() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, 2);
    PolicyDefinition policyDefinition2 = policyDefinition(POLICY_ID_2, POLICY_TEMPLATE_KEY_2, 1);

    server.deployPolicy(policyDefinition);

    assertEquals("Second Policy not correctly applied", POLICY_PAYLOAD, request.get().asString());

    server.deployPolicy(policyDefinition2);

    assertEquals("Second Policy not correctly applied", POLICY_PAYLOAD + POLICY_PAYLOAD_2, request.get().asString());

    assertThat(policyStore.load(), hasSize(2));
    assertThat(policyStore.load(), hasItems(policyDefinition, policyDefinition2));

    checkPhysicalFiles(policyDefinition);
    checkPhysicalFiles(policyDefinition2);
  }

  @Test
  public void deployPoliciesToDifferentApis() {
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, API_INJECTION_TEMPLATE_ID, EMPTY_API_KEY, null, 1,
                             new PolicyConfiguration(emptyMap()));
    PolicyDefinition policyDefinition2 =
        new PolicyDefinition(POLICY_ID_2, API_INJECTION_TEMPLATE_ID, SET_PAYLOAD_API_KEY, null, 1,
                             new PolicyConfiguration(emptyMap()));
    HttpRequest setPayloadRequest = request(httpPort, "/api/set-payload");

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    assertEquals("First Policy not correctly applied", EMPTY_API_KEY.id().toString(), request.get().asString());
    assertEquals("Second Policy not correctly applied", SET_PAYLOAD_API_KEY.id().toString(),
                 setPayloadRequest.get().asString());

    checkPhysicalFiles(policyDefinition);
    checkPhysicalFiles(policyDefinition2);
  }

  @Test
  public void removePolicy() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY);

    server.deployPolicy(policyDefinition);

    assertEquals("Policy not correctly applied", POLICY_PAYLOAD, request.get().asString());

    server.undeployPolicy(policyDefinition.getName());

    assertEquals("Policy not correctly un-applied", "", request.get().asString());

    assertThat(policyStore.load(), empty());

  }

  @Test
  public void removePolicyKeepsTheOthers() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, 2);
    PolicyDefinition policyDefinition2 = policyDefinition(POLICY_ID_2, POLICY_TEMPLATE_KEY_2, 1);

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition2);

    assertEquals("Policies not correctly applied", POLICY_PAYLOAD + POLICY_PAYLOAD_2, request.get().asString());

    server.undeployPolicy(policyDefinition.getName());

    assertEquals("Policies not correctly applied", POLICY_PAYLOAD_2, request.get().asString());

    assertThat(policyStore.load(), hasSize(1));
    assertThat(policyStore.load().get(0), is(policyDefinition2));
  }

  @Test
  public void removeNotDeployedPolicy() {
    // It does not fail
    server.undeployPolicy(POLICY_ID);
  }

  @Test
  public void deploySamePolicyTwiceThrowsError() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY);

    server.deployPolicy(policyDefinition);
    server.deployPolicy(policyDefinition);
  }

  @Test
  public void deployInvalidXmlPolicy() {
    PolicyDefinition policyDefinition = policyDefinition(INVALID_XML_TEMPLATE_ID);

    server.deployPolicy(policyDefinition);

    assertThat(policyStore.load(), hasSize(1));
    assertThat(policyStore.load().get(0), is(policyDefinition));
    assertThat(storeFiles.listPolicyDeploymentFailures(), hasSize(1));
  }

  @Test
  public void deployPolicyWithMissingTemplate() {
    PolicyTemplateKey policyTemplateKey = new PolicyTemplateKey("missing", "template", "1.0");
    PolicyDefinition policyDefinition = policyDefinition(policyTemplateKey);

    PolicyTemplate policyTemplate =
        new PolicyTemplate(policyTemplateKey, new File("missingFile"), new ValidPolicySpecification(),
                           new PolicyTemplateDescriptor("missing.template"));

    server.deployPolicy(new Policy(policyTemplate, policyDefinition, "resolvedTemplate"));

    assertThat(policyStore.load(), hasSize(1));
    assertThat(policyStore.load().get(0), is(policyDefinition));
    assertThat(storeFiles.listPolicyDeploymentFailures(), hasSize(1));
  }

  @Test
  public void isWsdlEndpointType() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY_3);

    server.deployPolicy(policyDefinition);
    assertThat(request.get().asString(), is("Other type of endpoint"));
  }

  @Test
  public void notWsdlEndpointType() {
    PolicyDefinition policyDefinition = wsdlPolicyDefinition(POLICY_TEMPLATE_KEY_3);

    server.deployPolicy(policyDefinition);
    assertThat(request.get().asString(), is("WSDL Endpoint Type"));
  }

  @Test
  public void deployInvalidAtInitPolicy() {
    PolicyDefinition policyDefinition = policyDefinition(INVALID_AT_INIT_TEMPLATE_ID);

    server.deployPolicy(policyDefinition);

    assertThat(policyStore.load(), hasSize(1));
    assertThat(policyStore.load().get(0), is(policyDefinition));
    assertThat(storeFiles.listPolicyDeploymentFailures(), hasSize(1));
  }

  private static FakeGatewayInstallation installation() {
    installation = builder()
        .withApplications(buildTestApplication(APP, "mule-config-http.xml"))
        .withPolicyTemplates(buildTestPolicyTemplate(POLICY_TEMPLATE_KEY, "policies/policy-deployment-policy.xml"),
                             buildTestPolicyTemplate(POLICY_TEMPLATE_KEY_2, "policies/policy-deployment-policy-2.xml"),
                             buildTestPolicyTemplate(POLICY_TEMPLATE_KEY_3, "policies/policy-is-wsdl-endpoint.xml"),
                             buildTestPolicyTemplate(INVALID_XML_TEMPLATE_ID,
                                                     "policies/policy-deployment-invalid-xml-policy.xml"),
                             buildTestPolicyTemplate(EMPTY_TEMPLATE_ID, "policies/policy-deployment-empty-policy.xml"),
                             buildTestPolicyTemplate(WITHOUT_EXECUTE_NEXT_TEMPLATE_ID,
                                                     "policies/policy-deployment-without-execute-next-policy.xml"),
                             buildTestPolicyTemplate(MULTIPLE_EXECUTE_NEXT_TEMPLATE_ID,
                                                     "policies/policy-deployment-multiple-execute-next-policy.xml"),
                             buildTestPolicyTemplate(INVALID_AT_INIT_TEMPLATE_ID,
                                                     "policies/policy-deployment-invalid-at-init-policy.xml"),
                             buildTestPolicyTemplate(API_INJECTION_TEMPLATE_ID,
                                                     "policies/policy-deployment-api-injection.xml"))
        .gateKeeperDisabled()
        .build();
    return installation;
  }

  private PolicyDefinition wsdlPolicyDefinition(PolicyTemplateKey templateKey) {
    return new PolicyDefinition(POLICY_ID, templateKey, EMPTY_API_KEY, null, 1,
                                new PolicyConfiguration(of("isWsdlEndpoint", "true")));
  }

  private PolicyDefinition policyDefinition(String id, PolicyTemplateKey templateKey, int order) {
    return new PolicyDefinition(id, templateKey, EMPTY_API_KEY, null, order, new PolicyConfiguration(emptyMap()));
  }

  private void checkPhysicalFiles(PolicyDefinition policyDefinition) {
    File policyFolder = storeFiles.getPolicyFolder(policyDefinition.getName());
    assertThat(policyFolder.exists(), is(true));
    assertThat(policyFolder.listFiles(), arrayWithSize(2));
    assertThat(new File(policyFolder, POLICY_DEFINITION_JSON_FILE_NAME).exists(), is(true));
    assertThat(new File(policyFolder, POLICY_CONFIG_XML_FILE_NAME).exists(), is(true));
  }

}
