/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.test.policy.deployment;

import static com.google.common.collect.Lists.newArrayList;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.APP_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_PAYLOAD;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_PAYLOAD_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY_2;
import static com.mulesoft.anypoint.tests.http.ApacheHttpRequest.request;
import static com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation.builder;
import static com.mulesoft.anypoint.tita.TestDependencies.contractsExtensionDependency;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestApplication;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestDomain;
import static com.mulesoft.anypoint.tita.environment.artifact.ArtifactProvider.buildTestPolicyTemplate;
import static java.util.Collections.singletonMap;
import static java.util.stream.Collectors.toSet;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;

import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tck.junit4.rule.SystemProperty;

import com.mulesoft.anypoint.tests.http.HttpRequest;
import com.mulesoft.anypoint.tests.http.HttpResponse;
import com.mulesoft.anypoint.tests.http.SimpleHttpServerResponse;
import com.mulesoft.anypoint.tests.infrastructure.FakeGatewayServer;
import com.mulesoft.anypoint.tests.infrastructure.installation.FakeGatewayInstallation;
import com.mulesoft.anypoint.tests.rules.HttpServerRule;
import com.mulesoft.anypoint.tita.environment.api.artifact.ApiFinder;
import com.mulesoft.anypoint.tita.environment.api.artifact.ApplicationJar;
import com.mulesoft.anypoint.tita.environment.api.artifact.DomainJar;
import com.mulesoft.mule.runtime.gw.api.policy.HttpResourcePointcut;
import com.mulesoft.mule.runtime.gw.api.policy.PolicyTemplateKey;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;

import com.google.common.collect.ImmutableMap;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.junit.After;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.RuleChain;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(Parameterized.class)
public class PolicyOperationWithPointcutsDeploymentTestCase extends AbstractMuleTestCase {

  private static final String NON_HTTP_OPERATION_API = "with-non-http-operation";
  private static final String REQUESTER_API = "with-requester";
  private static final String BOTH_API = "with-both";
  private static final String MULTIPLE_REQUESTERS_API = "with-multiple-requesters";
  private static final String REQUESTER_IN_SUB_FLOW_API = "with-requester-in-sub-flow";
  private static final String ANNOTATED_LISTENER_API = "with-annotated-listener";
  private static final String ANNOTATED_REQUESTER_API = "with-annotated-requester";
  private static final String ALTERNATE_ANNOTATED_REQUESTERS_API = "with-alternate-annotated-requesters";

  private static final String BACKEND_BODY = "body";

  private static DynamicPort appPort1 = new DynamicPort("port");
  private static DynamicPort domainPort = new DynamicPort("domainPort");

  private static SystemProperty policyPayload = new SystemProperty("policyPayload", POLICY_PAYLOAD);
  private static SystemProperty policyPayload2 = new SystemProperty("policyPayload2", POLICY_PAYLOAD_2);

  private static DomainJar domain = buildTestDomain("domain", "mule-domain-http-3.xml");
  private static ApplicationJar application1 =
      buildTestApplication(APP, "mule-config-with-operations.xml", contractsExtensionDependency());
  private static ApplicationJar application2 =
      buildTestApplication(APP_2, "mule-config-with-operations-domain.xml", domain, contractsExtensionDependency());

  private static FakeGatewayInstallation installation =
      builder()
          .withDomains(domain)
          .withApplications(application1,
                            application2)
          .withPolicyTemplates(buildTestPolicyTemplate(POLICY_TEMPLATE_KEY,
                                                       "policies/policy-operation-with-pointcuts-policy.xml"),
                               buildTestPolicyTemplate(POLICY_TEMPLATE_KEY_2,
                                                       "policies/policy-operation-scope-add-request-headers-policy.xml"))
          .gateKeeperDisabled()
          .offline()
          .build();

  private static HttpServerRule backendServer = new HttpServerRule("backendPort");

  @ClassRule
  public static RuleChain ruleChain = RuleChain.outerRule(appPort1)
      .around(domainPort)
      .around(policyPayload)
      .around(policyPayload2)
      .around(backendServer)
      .around(installation);

  private final ApiFinder apiFinder;

  private HttpRequest nonHttpOperationRequest;
  private HttpRequest requesterRequest;
  private HttpRequest bothRequest;
  private HttpRequest multipleRequestersRequest;
  private HttpRequest requesterInSubFlow;
  private HttpRequest withAnnotatedListener;
  private HttpRequest withAnnotatedRequester;
  private HttpRequest withAlternateAnnotatedRequester;

  private FakeGatewayServer server = installation.getServer();

  public PolicyOperationWithPointcutsDeploymentTestCase(String name, DynamicPort port, ApplicationJar application) {
    apiFinder = new ApiFinder(application.getAppConfig());
    nonHttpOperationRequest = request(port, "/api/non-http-operation");
    requesterRequest = request(port, "/api/requester");
    bothRequest = request(port, "/api/both");
    multipleRequestersRequest = request(port, "/api/multiple-requesters");
    requesterInSubFlow = request(port, "/api/requester-in-sub-flow");
    withAnnotatedListener = request(port, "/api/with-annotated-listener");
    withAnnotatedRequester = request(port, "/api/with-annotated-requester");
    withAlternateAnnotatedRequester = request(port, "/api/with-alternate-annotated-requesters");
  }

  @Parameterized.Parameters(name = "Application {0}")
  public static Object[][] parameters() {
    return new Object[][] {
        {"without domain", appPort1, application1},
        {"with domain", domainPort, application2},
    };
  }

  @Before
  public void setUp() {
    SimpleHttpServerResponse response = SimpleHttpServerResponse.builder().body(BACKEND_BODY).build();
    backendServer.getHttpServer().setResponse(response);
  }

  @After
  public void tearDown() {
    server.removeAllPoliciesAndContext();
  }

  @Test
  public void apiPointcutMatchesRequester() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY, REQUESTER_API);

    server.deployPolicy(policyDefinition);

    assertThat(requesterRequest.get().asString(), is(BACKEND_BODY + POLICY_ID));
  }

  @Test
  public void apiPointcutNotMatchesToRequesterWhenApiDoesNotMatch() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY, BOTH_API);

    server.deployPolicy(policyDefinition);

    assertThat(requesterRequest.get().asString(), is(BACKEND_BODY));
  }

  @Test
  public void apiPointcutNotMatchesToNonRequesterOperation() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY, NON_HTTP_OPERATION_API);

    server.deployPolicy(policyDefinition);

    assertThat(nonHttpOperationRequest.get().asString(), is("non-http-operation"));
  }

  @Test
  public void apiPointcutMatchesOnlyRequesterWhenMultipleOperationsPresent() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY, BOTH_API);

    server.deployPolicy(policyDefinition);

    assertThat(bothRequest.get().asString(), is(BACKEND_BODY + POLICY_ID));
  }

  @Test
  public void apiPointcutMatchesEveryRequester() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY, MULTIPLE_REQUESTERS_API);

    server.deployPolicy(policyDefinition);

    assertThat(multipleRequestersRequest.get().asString(), is(BACKEND_BODY + POLICY_ID + BACKEND_BODY + POLICY_ID));
  }

  @Test
  public void resourcePointcutMatchesRequester() {
    PolicyDefinition policyDefinition =
        policyDefinition(POLICY_TEMPLATE_KEY, REQUESTER_API, new HttpResourcePointcut(".*", "GET"));

    server.deployPolicy(policyDefinition);

    assertThat(requesterRequest.get().asString(), is(BACKEND_BODY + POLICY_ID));
    assertThat(requesterRequest.post().asString(), is(BACKEND_BODY));
  }

  @Test
  public void apiPointcutMatchesRequesterInSubFlow() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY, REQUESTER_IN_SUB_FLOW_API);

    server.deployPolicy(policyDefinition);

    assertThat(requesterInSubFlow.get().asString(), is(BACKEND_BODY + POLICY_ID));
  }

  @Test
  public void httpPointcutNotMatchesToAnnotatedRequester() {
    Map<String, Object> configData = ImmutableMap.of("propagationEnabled", "true",
                                                     "headerName", "policyHeader1",
                                                     "headerValue", "policyHeaderValue1",
                                                     "headerName2", "policyHeader2",
                                                     "headerValue2", "policyHeaderValue2");
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY_2, ANNOTATED_REQUESTER_API, configData);

    server.deployPolicy(policyDefinition);

    HttpResponse response = withAnnotatedRequester.get();

    assertThat(response.statusCode(), is(200));
    Set<String> backendRequestHeaders = backendServer.getHttpServer().getLastHttpRequest().getHeaders().keySet().stream()
        .map(k -> k.toLowerCase()).collect(toSet());
    assertThat(backendRequestHeaders.contains("policyheader1"), is(false));
    assertThat(backendRequestHeaders.contains("policyheader2"), is(false));
  }

  @Test
  public void httpPointcutNotMatchesToAlternatedAnnotatedRequesters() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY, ALTERNATE_ANNOTATED_REQUESTERS_API);

    server.deployPolicy(policyDefinition);

    HttpResponse response = withAlternateAnnotatedRequester.get();

    assertThat(response.statusCode(), is(200));
    assertThat(response.asString(), is(BACKEND_BODY + BACKEND_BODY + POLICY_ID));
  }

  @Test
  public void httpPointcutMatchesAnnotatedListener() {
    PolicyDefinition policyDefinition = policyDefinition(POLICY_TEMPLATE_KEY, ANNOTATED_LISTENER_API);

    server.deployPolicy(policyDefinition);

    assertThat(withAnnotatedListener.get().asString(), is(BACKEND_BODY + POLICY_ID));
  }

  private PolicyDefinition policyDefinition(PolicyTemplateKey templateKey, String apiFlow) {
    return policyDefinition(templateKey, apiFlow, new HashMap<>());
  }

  private PolicyDefinition policyDefinition(PolicyTemplateKey templateKey, String apiFlow,
                                            HttpResourcePointcut resourcePointcut) {
    return new PolicyDefinition(POLICY_ID, templateKey, apiFinder.find(apiFlow), newArrayList(resourcePointcut), 1,
                                new PolicyConfiguration(singletonMap("policyId", POLICY_ID)));
  }

  private PolicyDefinition policyDefinition(PolicyTemplateKey templateKey, String apiFlow, Map<String, Object> configData) {
    Map<String, Object> effectiveConfigData =
        ImmutableMap.<String, Object>builder().putAll(configData).put("policyId", POLICY_ID).build();
    return new PolicyDefinition(POLICY_ID, templateKey, apiFinder.find(apiFlow), null, 1,
                                new PolicyConfiguration(effectiveConfigData));
  }

}
